﻿using System;
using System.Diagnostics.CodeAnalysis;
using System.Linq;
using System.Security.Cryptography;
using System.Text;

namespace Magify
{
	/// <summary>
	/// Helper methods for working with <see cref="Guid"/>.
	/// </summary>
	internal static class GuidUtility
	{
		/// <summary>
		/// Creates a name-based UUID using the algorithm from RFC 4122 §4.3.
		/// </summary>
		/// <param name="namespaceId">The ID of the namespace.</param>
		/// <param name="name">The name (within that namespace).</param>
		/// <returns>A UUID derived from the namespace and name.</returns>
		public static Guid Create(Guid namespaceId, [JetBrains.Annotations.NotNull] string name)
		{
			// convert the name to a sequence of octets (as defined by the standard or conventions of its namespace) (step 3)
			// ASSUME: UTF-8 encoding is always appropriate
			return Create(namespaceId, Encoding.UTF8.GetBytes(name));
		}

		/// <summary>
		/// Creates a name-based UUID using the algorithm from RFC 4122 §4.3.
		/// </summary>
		/// <param name="namespaceId">The ID of the namespace.</param>
		/// <param name="nameBytes">The name (within that namespace).</param>
		/// <returns>A UUID derived from the namespace and name.</returns>
		[SuppressMessage("Security", "CA5350:Do Not Use Weak Cryptographic Algorithms", Justification = "Per spec.")]
		[SuppressMessage("Security", "CA5351:Do Not Use Broken Cryptographic Algorithms", Justification = "Per spec.")]
		public static Guid Create(Guid namespaceId, [JetBrains.Annotations.NotNull] byte[] nameBytes)
        {
            // https://datatracker.ietf.org/doc/html/rfc4122#section-4.3

            // structure:
            // 00000000-0000-0000-0000-000000000000
            // 00000000-____-____-____-____________ - time_low
            // ________-0000-____-____-____________ - time_mid
            // ________-____-0000-____-____________ - time_hi_and_version
            // ________-____-____-0000-____________ - clock_seq_hi_and_res (clock_seq_low) - not in this case
            // ________-____-____-____-000000000000 - node

            // convert the namespace UUID to network order (step 3)
			var namespaceBytes = namespaceId.ToByteArray();
            // Unity builds target only little-endian architectures, so `BitConverter.IsLittleEndian` is always true.
            // However, this swap is NOT about CPU endianness — it's to convert between the .NET Guid internal layout
            // (which stores the first 3 fields in little-endian) and RFC 4122 network byte order (big-endian for all fields).
            // We must always swap here to get correct RFC-compliant bytes for hashing.
			SwapByteOrder(namespaceBytes);

			// compute the hash of the namespace ID concatenated with the name (step 4)
			var data = namespaceBytes.Concat(nameBytes).ToArray();
			byte[] hash;
            using (var algorithm = SHA1.Create())
            {
                hash = algorithm!.ComputeHash(data);
            }

			// most bytes from the hash are copied straight to the bytes of the new GUID (steps 5-7, 9, 11-12)
			var newGuid = new byte[16];
			Array.Copy(hash, 0, newGuid, 0, 16);

			// set the four most significant bits (bits 12 through 15) of the time_hi_and_version field to the appropriate 4-bit version number from Section 4.1.3 (step 8)
			newGuid[6] = (byte) ((newGuid[6] & 0x0F) | 0x50); // 0x50 == 0b0101 == 5 (version) << 4 (bits) And 4.1.3 says to set it

			// set the two most significant bits (bits 6 and 7) of the clock_seq_hi_and_reserved to zero and one, respectively (step 10)
			newGuid[8] = (byte) ((newGuid[8] & 0x3F) | 0x80);

			// convert the resulting UUID to local byte order (step 13)
			SwapByteOrder(newGuid);
			return new Guid(newGuid);
		}

		// Converts a GUID (expressed as a byte array) to/from network order (MSB-first).
		private static void SwapByteOrder([JetBrains.Annotations.NotNull] byte[] guid)
		{
            // time_low
			SwapBytes(guid, 0, 3);
			SwapBytes(guid, 1, 2);
            // time_mid
			SwapBytes(guid, 4, 5);
            // time_hi_and_version
			SwapBytes(guid, 6, 7);
            // clock_seq + node: leave a[8..15] unchanged
		}

		private static void SwapBytes([JetBrains.Annotations.NotNull] byte[] guid, int left, int right)
		{
			(guid[left], guid[right]) = (guid[right], guid[left]);
        }
	}
}