﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.CompilerServices;
using System.Threading;
using System.Threading.Tasks;
using JetBrains.Annotations;
using UnityEngine;
using UnityEngine.Networking;
using UnityEngine.UI;

namespace Magify
{
    internal static class ExtensionMethods
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get();

        internal static TaskAwaiter GetAwaiter(this AsyncOperation asyncOp)
        {
            var tcs = new TaskCompletionSource<object>();
            asyncOp.completed += _ => { tcs.SetResult(null); };
            return ((Task)tcs.Task).GetAwaiter();
        }

        public static TValue GetValueOrDefault<TKey, TValue>(this IDictionary<TKey, TValue> dictionary, TKey key)
        {
            return dictionary.TryGetValue(key, out var value) ? value : default;
        }

        public static bool TryGetValue<T>(this IReadOnlyDictionary<string, object> map, string key, out T value)
        {
            value = default;
            if (!map.TryGetValue(key, out var valueObject))
            {
                return false;
            }
            try
            {
                if (valueObject is T valueTyped)
                {
                    value = valueTyped;
                }
                else
                {
                    value = (T)Convert.ChangeType(valueObject, typeof(T));
                }

                return true;
            }
            catch
            {
                return false;
            }
        }

        public static double GetDouble(this IReadOnlyDictionary<string, object> dict, string key, double defaultValue = default)
        {
            if (!dict.TryGetValue(key, out var countRaw))
            {
                return defaultValue;
            }
            switch (countRaw)
            {
                case int intValue:
                    return intValue;
                case float floatValue:
                    return floatValue;
                case double doubleValue:
                    return doubleValue;
                default:
                    try
                    {
                        return Convert.ToDouble(countRaw);
                    }
                    catch
                    {
                        return defaultValue;
                    }
            }
        }

        public static int AddRange<T>([NotNull] this ICollection<T> source, [CanBeNull, ItemCanBeNull] IEnumerable<T> items)
        {
            var count = 0;
            if (items == null)
            {
                return count;
            }

            foreach (var item in items)
            {
                count++;
                source.Add(item);
            }
            return count;
        }

        public static void ForEach<T>(this IEnumerable<T> list, Action<T> predicate)
        {
            foreach (var item in list)
            {
                predicate(item);
            }
        }

        public static GameObject GetMainGameObject(this AssetBundle bundle)
        {
            var loadAllAssets = bundle.LoadAllAssets<GameObject>();
            var content = loadAllAssets.FirstOrDefault();
            return content;
        }

        internal static void SetAlpha(this Image image, float a)
        {
            var color = image.color;
            color.a = a;
            image.color = color;
        }

        internal static void SetAlpha(this Button button, float a)
        {
            button.GetComponent<Image>().SetAlpha(a);
        }

        internal static Color ToColor(this string htmlColor)
        {
            if (!TryParseHtmlString(htmlColor, out var color))
            {
                color = Color.gray;
            }
            return color;
        }

        internal static bool TryParseHtmlString(string hex, out Color color)
        {
            color = Color.magenta;
            try
            {
                hex = hex.Replace("#", string.Empty);
                if (hex.Length != 6 && hex.Length != 8)
                {
                    return false;
                }
                var r = (byte)Convert.ToUInt32(hex.Substring(0, 2), 16);
                var g = (byte)Convert.ToUInt32(hex.Substring(2, 2), 16);
                var b = (byte)Convert.ToUInt32(hex.Substring(4, 2), 16);
                var a = hex.Length == 8
                    ? (byte)Convert.ToUInt32(hex.Substring(6, 2), 16)
                    : byte.MaxValue;
                color = new Color32(r, g, b, a);
                return true;
            }
            catch (Exception e)
            {
                Debug.LogError($"[Magify]: Can't parse color from html: {hex}. Error: {e.Message}");
                return false;
            }
        }

        /// <summary>
        /// Wraps an AsyncOperation with a CancellationToken to allow cancellation of the operation.
        /// </summary>
        /// <param name="requestAsyncOperation">The UnityWebRequestAsyncOperation to be wrapped.</param>
        /// <param name="cancellationToken">The CancellationToken to use for cancellation.</param>
        /// <returns>A Task that represents the completion of the AsyncOperation or the cancellation through the CancellationToken.</returns>
        [NotNull]
        internal static Task WithCancellation([NotNull] this UnityWebRequestAsyncOperation requestAsyncOperation, CancellationToken cancellationToken)
        {
            if (cancellationToken.IsCancellationRequested)
            {
                requestAsyncOperation.webRequest!.Abort();
                return Task.FromCanceled(cancellationToken);
            }

            var tcs = new TaskCompletionSource<object>();
            requestAsyncOperation.completed += _ => tcs.TrySetResult(null);
            cancellationToken.Register(() =>
            {
                if (tcs.TrySetCanceled(cancellationToken))
                {
                    try
                    {
                        _logger.Log("Aborting unity web request because cancellation token is cancelled");
                        requestAsyncOperation.webRequest!.Abort();
                    }
                    catch (Exception e)
                    {
                        _logger.LogWarning($"Magify caught an exception while aborting the web request: {e}");
                    }
                }
            });
            return tcs.Task!;
        }

        internal static async Task<bool> SuppressCancellationThrow(this Task task)
        {
            var taskStatus = task.Status;
            return taskStatus switch
            {
                TaskStatus.RanToCompletion => false,
                TaskStatus.Canceled => true,
                _ => await task.ContinueWith(_ => task.Status == TaskStatus.Canceled)
            };
        }

        internal static Task RepeatWhile(this Func<Task> task, Func<bool> endCondition, CancellationToken token, ExponentialBackoff backoff)
        {
            if (token.IsCancellationRequested)
            {
                return Task.FromCanceled(token);
            }
            var promise = new TaskCompletionSource<object>();
            token.Register(() => promise.TrySetCanceled(token));
            routine();
            return promise.Task;

            async void routine()
            {
                while (true)
                {
                    await task();
                    if (token.IsCancellationRequested)
                    {
                        promise.TrySetCanceled(token);
                        return;
                    }

                    if (endCondition())
                    {
                        promise.TrySetResult(null);
                        break;
                    }

                    await Task.Delay(TimeSpan.FromMilliseconds(backoff.NextDelay()), token);
                }
            }
        }
    }
}