using System;
using JetBrains.Annotations;

namespace Magify
{
    /// <summary>
    /// Thread-safe exponential backoff.
    /// </summary>
    internal class ExponentialBackoff
    {
        [NotNull]
        private readonly object _lock = new();
        private readonly ulong _delayMilliseconds;
        private readonly ulong _maxDelayMilliseconds;
        private uint _retries;
        private uint _pow;

        public uint Retries
        {
            get
            {
                lock (_lock)
                {
                    return _retries;
                }
            }
        }

        public ExponentialBackoff(ulong delayMilliseconds = 1000, ulong maxDelayMilliseconds = 64000)
        {
            _delayMilliseconds = delayMilliseconds;
            _maxDelayMilliseconds = maxDelayMilliseconds;
            Reset();
        }

        public void Reset()
        {
            lock (_lock)
            {
                _retries = 0;
                _pow = 1;
            }
        }

        /// <returns>
        /// Milliseconds delay
        /// </returns>
        public ulong NextDelay()
        {
            lock (_lock)
            {
                _retries += 1;
                if (_retries < 31)
                {
                    _pow = _pow << 1;
                }
                return Math.Min(_delayMilliseconds * (_pow - 1) / 2, _maxDelayMilliseconds);
            }
        }
    }
}