using System;
using System.Collections.Generic;
using System.Threading;
using JetBrains.Annotations;
using Magify.Rx;
using UnityEngine.Pool;

namespace Magify
{
    public class CompositeCancellationTokenSource : ICancelable
    {
        [NotNull]
        private readonly CancellationTokenSource _cancellationTokenSource = new();
        [NotNull]
        private readonly object _lock = new();
        [NotNull]
        private readonly Dictionary<CancellationToken, CancellationTokenRegistration> _registrationTokens =
                     DictionaryPool<CancellationToken, CancellationTokenRegistration>.Get() ?? new();
        private volatile bool _isDisposed;

        public bool IsCancellationRequested => _cancellationTokenSource.IsCancellationRequested;
        public CancellationToken Token => _cancellationTokenSource.Token;

        public bool IsDisposed
        {
            get
            {
                lock (_lock)
                {
                    return _isDisposed;
                }
            }
        }

        public CompositeCancellationTokenSource()
        {
        }

        public CompositeCancellationTokenSource(CancellationToken initialToken)
        {
            if (initialToken.IsCancellationRequested || !initialToken.CanBeCanceled)
            {
                return;
            }
            Register(initialToken);
        }

        public void Register(CancellationToken token)
        {
            if (token.IsCancellationRequested || !token.CanBeCanceled)
            {
                return;
            }
            lock (_lock)
            {
                ThrowIfDisposedThreadUnsafe();
                if (token.IsCancellationRequested || !token.CanBeCanceled)
                {
                    return;
                }
                var registration = token.Register(cancelledHandler);
                _registrationTokens.Add(token, registration);
            }

            void cancelledHandler() => TokenCancelledHandler(token);
        }

        private void TokenCancelledHandler(CancellationToken token)
        {
            bool shouldCancel;
            lock (_lock)
            {
                if (_isDisposed)
                {
                    return;
                }
                if (_registrationTokens.Remove(token, out var registration))
                {
                    registration.Dispose();
                }
                shouldCancel = _registrationTokens.Count == 0;
            }

            if (shouldCancel)
            {
                Cancel();
            }
        }

        public void Cancel()
        {
            lock (_lock)
            {
                ThrowIfDisposedThreadUnsafe();
                _cancellationTokenSource.Cancel();
            }
        }

        public void CancelAfter(int millisecondsDelay)
        {
            lock (_lock)
            {
                ThrowIfDisposedThreadUnsafe();
                _cancellationTokenSource.CancelAfter(millisecondsDelay);
            }
        }

        public void CancelAfter(TimeSpan delay)
        {
            lock (_lock)
            {
                ThrowIfDisposedThreadUnsafe();
                _cancellationTokenSource.CancelAfter(delay);
            }
        }

        public void Dispose()
        {
            lock (_lock)
            {
                if (_isDisposed)
                {
                    return;
                }

                _isDisposed = true;

                foreach (var registration in _registrationTokens.Values)
                    registration.Dispose();
                _registrationTokens.Clear();
                DictionaryPool<CancellationToken, CancellationTokenRegistration>.Release(_registrationTokens);
                _cancellationTokenSource.Dispose();
            }
        }

        /// <inheritdoc cref="MagifyDocs.NoThreadSync"/>
        private void ThrowIfDisposedThreadUnsafe()
        {
            if (IsDisposed)
            {
                throw new ObjectDisposedException(nameof(CompositeCancellationTokenSource));
            }
        }
    }
}