﻿using System;
using System.Collections.Generic;
using System.Linq;
using JetBrains.Annotations;

namespace Magify
{
    public static class CollectionExtensions
    {
        public delegate void ActionSpan(ref ReadOnlySpan<char> key);

        private static class GenericMap<TKey, TValue>
        {
            private static Dictionary<TKey, TValue> _map;

            [NotNull]
            public static Dictionary<TKey, TValue> Get(int size = default)
            {
                if (_map == null)
                {
                    _map = new Dictionary<TKey, TValue>(size);
                }
                else
                {
                    _map.Clear();
                }
                return _map;
            }
        }

        public static void Split(this string key, char separator, ActionSpan action)
        {
            key.Split(separator, StringSplitOptions.None, action);
        }

        public static void Split(this string key, char separator, StringSplitOptions options, ActionSpan action)
        {
            var source = key.AsSpan();
            var position = 0;

            while (position < key.Length)
            {
                var tail = source[position..];
                var foundIndex = tail.IndexOf(separator);
                var next = foundIndex != -1
                    ? tail[..foundIndex]
                    : tail;

                if (options == StringSplitOptions.None || next.Length > 0)
                {
                    action(ref next);
                }

                position += next.Length + 1;
            }
        }

        /// <summary>
        /// Check for Equality in ANY order (Fast). Supports null
        /// </summary>
        public static bool EqualsInAnyOrder<T>(this IReadOnlyCollection<T> list1, IReadOnlyCollection<T> list2)
        {
            if (ReferenceEquals(list1, list2)) return true;
            if (list1 == null || list2 == null) return false;
            if (list1.Count != list2.Count) return false;

            var counts = GenericMap<T, int>.Get(list1.Count);
            var nullCount = 0;

            foreach (var value in list1)
            {
                if (value == null)
                {
                    nullCount++;
                    continue;
                }

                if (counts.TryGetValue(value, out var count))
                {
                    counts[value] = count + 1;
                }
                else
                {
                    counts[value] = 1;
                }
            }

            var ok = true;
            foreach (var value in list2)
            {
                if (value == null)
                {
                    nullCount--;
                    continue;
                }

                if (counts.TryGetValue(value, out var c))
                {
                    counts[value] = c - 1;
                }
                else
                {
                    ok = false;
                    break;
                }
            }
            return ok && nullCount == 0 && counts.Values.All(c => c == 0);
        }

        /// <summary>
        /// Check for Equality in ANY order (Fast). Supports null
        /// </summary>
        public static bool EqualsInAnyOrder<T>([NotNull] this IEnumerable<T> list1, int count1, [NotNull] IEnumerable<T> list2, int count2)
        {
            if (ReferenceEquals(list1, list2)) return true;
            if (count1 != count2) return false;

            var counts = GenericMap<T, int>.Get(count1);
            var nullCount = 0;

            foreach (var value in list1)
            {
                if (value is null)
                {
                    nullCount++;
                    continue;
                }

                if (counts.TryGetValue(value, out var count))
                {
                    counts[value] = count + 1;
                }
                else
                {
                    counts[value] = 1;
                }
            }

            var ok = true;
            foreach (var value in list2)
            {
                if (value is null)
                {
                    nullCount--;
                    continue;
                }

                if (counts.TryGetValue(value, out var c))
                {
                    counts[value] = c - 1;
                }
                else
                {
                    ok = false;
                    break;
                }
            }
            return ok && nullCount == 0 && counts.Values.All(c => c == 0);
        }
    }
}