using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using JetBrains.Annotations;

namespace Magify
{
    public partial class BinaryStorage
    {
        static partial void LockFilePathInEditor(string filePath);
        static partial void ThrowIfFilePathLocked(string filePath);
        static partial void UnlockFilePathInEditor(string filePath);
        static partial void SaveAsJsonForDebug(string filePath, List<Section> sections);

        public static BinaryStorage Get([NotNull] string filePath)
        {
            return Construct(filePath)
                .AddPrimitiveTypes()
                .Build();
        }

        [NotNull]
        public static Builder Construct([NotNull] string filePath)
        {
            ThrowIfFilePathLocked(filePath);
            LockFilePathInEditor(filePath);
            return new Builder(filePath);
        }

        internal static void Delete([NotNull] string storagePath)
        {
            if (File.Exists(storagePath))
            {
                File.Delete(storagePath);
            }
        }

        public class Builder
        {
            [NotNull]
            private readonly string _filePath;
            [NotNull, ItemNotNull]
            private readonly List<Section> _sections = new();

            private bool _throwOnLoadingError;

            public Builder([NotNull] string filePath)
            {
                _filePath = filePath;
            }

            [NotNull]
            public Builder AddPrimitiveTypes()
            {
                return AddEntry(BinaryTypeBoolean.Shared)
                    .AddEntry(BinaryTypeChar.Shared)
                    .AddEntry(BinaryTypeByte.Shared)
                    .AddEntry(BinaryTypeSByte.Shared)
                    .AddEntry(BinaryTypeInt16.Shared)
                    .AddEntry(BinaryTypeUInt16.Shared)
                    .AddEntry(BinaryTypeInt32.Shared)
                    .AddEntry(BinaryTypeUInt32.Shared)
                    .AddEntry(BinaryTypeInt64.Shared)
                    .AddEntry(BinaryTypeUInt64.Shared)
                    .AddEntry(BinaryTypeSingle.Shared)
                    .AddEntry(BinaryTypeDouble.Shared)
                    .AddEntry(BinaryTypeDecimal.Shared)
                    .AddEntry(BinaryTypeString.Shared)
                    .AddEntry(BinaryTypeDateTime.Shared)
                    .AddEntry(BinaryTypeVector2.Shared)
                    .AddEntry(BinaryTypeVector3.Shared);
            }

            [NotNull]
            public Builder SupportNullable<T>()
                where T: struct
            {
                return AddEntry(BinaryTypeNullable<T>.Shared);
            }

            [NotNull]
            public Builder SupportEnum<T>(bool useFullName = false)
                where T : unmanaged, Enum
            {
                var enumType = typeof(T);
                var underlyingType = Enum.GetUnderlyingType(enumType);
                switch (underlyingType)
                {
                    case not null when underlyingType == typeof(byte):
                        AddEntry(new BinaryEnumHandler<T, byte>(BinaryTypeByte.Shared, useFullName));
                        break;
                    case not null when underlyingType == typeof(sbyte):
                        AddEntry(new BinaryEnumHandler<T, sbyte>(BinaryTypeSByte.Shared, useFullName));
                        break;
                    case not null when underlyingType == typeof(short):
                        AddEntry(new BinaryEnumHandler<T, short>(BinaryTypeInt16.Shared, useFullName));
                        break;
                    case not null when underlyingType == typeof(ushort):
                        AddEntry(new BinaryEnumHandler<T, ushort>(BinaryTypeUInt16.Shared, useFullName));
                        break;
                    case not null when underlyingType == typeof(int):
                        AddEntry(new BinaryEnumHandler<T, int>(BinaryTypeInt32.Shared, useFullName));
                        break;
                    case not null when underlyingType == typeof(uint):
                        AddEntry(new BinaryEnumHandler<T, uint>(BinaryTypeUInt32.Shared, useFullName));
                        break;
                    case not null when underlyingType == typeof(long):
                        AddEntry(new BinaryEnumHandler<T, long>(BinaryTypeInt64.Shared, useFullName));
                        break;
                    case not null when underlyingType == typeof(ulong):
                        AddEntry(new BinaryEnumHandler<T, ulong>(BinaryTypeUInt64.Shared, useFullName));
                        break;
                    default:
                        throw new UnexpectedUnderlyingEnumTypeException(enumType, underlyingType);
                }
                return this;
            }

            [NotNull]
            public Builder SupportSetsOf<T>()
            {
                if (_sections.Any(isSet))
                {
                    return this;
                }

                if (_sections.Find(isItem) is not TypedSection<T> typedSection)
                {
                    throw new BinaryStorageDoesntSupportSetOfException(typeof(T));
                }

                return AddSection(new CollectionSection<T, ReactiveHashSet<T>>(typedSection.BinaryType));
                bool isSet(Section s) => s is CollectionSection<T, ReactiveHashSet<T>>;
                bool isItem(Section s) => s is TypedSection<T>;
            }

            public Builder SupportListsOf<T>()
            {
                if (_sections.Any(isList))
                {
                    return this;
                }

                if (_sections.Find(isItem) is not TypedSection<T> typedSection)
                {
                    throw new BinaryStorageDoesntSupportListOfException(typeof(T));
                }

                return AddSection(new CollectionSection<T, ReactiveList<T>>(typedSection.BinaryType));
                bool isList(Section s) => s is CollectionSection<T, ReactiveList<T>>;
                bool isItem(Section s) => s is TypedSection<T>;
            }

            [NotNull]
            public Builder SupportDictionariesOf<TKey, TValue>()
            {
                if (_sections.Any(isDictionary))
                {
                    return this;
                }

                var typedSectionKey = _sections.Find(isKey) as TypedSection<TKey>;
                var typedSectionValue = _sections.Find(isValue) as TypedSection<TValue>;
                if (typedSectionKey == null || typedSectionValue == null)
                {
                    throw new BinaryStorageDoesntSupportDictionaryOfException(typeof(TKey), typeof(TValue));
                }

                var binaryType = new BinaryTypesPair<TKey, TValue>(typedSectionKey.BinaryType, typedSectionValue.BinaryType);
                return AddSection(new CollectionSection<KeyValuePair<TKey, TValue>, ReactiveDictionary<TKey, TValue>>(binaryType));
                bool isDictionary(Section s) => s is CollectionSection<KeyValuePair<TKey, TValue>, ReactiveDictionary<TKey, TValue>>;
                bool isKey(Section s) => s is TypedSection<TKey>;
                bool isValue(Section s) => s is TypedSection<TValue>;
            }

            [NotNull]
            public Builder AddEntry<T>([NotNull] BinaryTypeHandler<T> binaryTypeHandler)
            {
                if (_sections.Any(isSectionOfType))
                {
                    throw new DuplicateBinaryStorageSectionException(typeof(T), Path.GetFileName(_filePath));
                }
                return AddSection(new TypedSection<T>(binaryTypeHandler));
                bool isSectionOfType(Section s) => s is TypedSection<T>;
            }

            public Builder ThrowOnLoadError()
            {
                _throwOnLoadingError = true;
                return this;
            }

            [NotNull]
            private Builder AddSection([NotNull] Section section)
            {
                var existing = _sections.Find(isSameSection);
                if (existing != null)
                {
                    throw new DuplicateBinaryStorageSectionNameException(section.Name, Path.GetFileName(_filePath));
                }
                _sections.Add(section);
                return this;
                bool isSameSection(Section s) => s.Name == section.Name;
            }

            [NotNull]
            public BinaryStorage Build()
            {
                var storage = new BinaryStorage(_filePath, _sections);
                var isLoaded = false;
                for (var i = 0; i < 3 && !isLoaded; i++)
                {
                    isLoaded = tryLoad(i != 0);
                }

                return storage;

                bool tryLoad(bool isRetry)
                {
                    try
                    {
                        try
                        {
                            storage.LoadDataFromDisk(Encoding.UTF8, _throwOnLoadingError);
                        }
                        catch
                        {
                            // In case of any error we have to try to load storage using old data format
                            // Remove this logic somewhere in 2025 or 2026

                            // Few things were changed:
                            // 1) Storage now uses UTF8 instead of Unicode
                            // https://github.com/app-craft/gandalf-unity-plugin/pull/407

                            // 2) Strings formatters now use UTF8 instead of Unicode
                            // https://github.com/app-craft/gandalf-unity-plugin/pull/407

                            // 3) DateTime now uses FromBinary and ToBinary methods instead of saving ticks
                            // https://github.com/app-craft/gandalf-unity-plugin/pull/389
                            // https://github.com/app-craft/gandalf-unity-plugin/pull/394

                            // Use legacy DateTime and string binary formatters (to load data, that`s in legacy format)
                            var previousDateTimeTypeHandler = setTypeHandlerOf(BinaryTypeDateTimeLegacy.Shared);
                            var previousStringTypeHandler = setTypeHandlerOf(new BinaryTypeString(Encoding.Unicode));

                            // Reload storage one more
                            storage.LoadDataFromDisk(Encoding.Unicode, false);

                            // Restore storage binary formatters (to save data (in future) in current format)
                            setTypeHandlerOf(previousDateTimeTypeHandler);
                            setTypeHandlerOf(previousStringTypeHandler);

                            /// <returns>Previous BinaryTypeHandler</returns>
                            BinaryTypeHandler<T> setTypeHandlerOf<T>(BinaryTypeHandler<T> source)
                            {
                                if (source == null) return null;
                                var index = _sections.FindIndex(isSectionOfType);
                                if (index != -1)
                                {
                                    var section = (TypedSection<T>)_sections[index];
                                    var result = section!.BinaryType;
                                    section.BinaryType = source;
                                    return result;
                                }
                                return null;
                                bool isSectionOfType(Section s) => s is TypedSection<T>;
                            }
                        }
                    }
                    catch (Exception e)
                    {
                        _logger.LogError(new MagifyFailedToLoadBinaryStorageException(_filePath, isRetry, e).Message);
                        return false;
                    }
                    return true;
                }
            }
        }
    }
}