using System;
using System.IO;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using UnityEngine;
using UnityEngine.Networking;
using Magify.Model;

namespace Magify
{
    internal readonly struct SetupDefaultContextJob : IMagifyWorkerJob
    {
        public string ConfigPath { get; }

        public SetupDefaultContextJob([NotNull] string configPath)
        {
            ConfigPath = configPath;
        }
    }

    internal class SetupDefaultContextWorker : IMagifyWorker<SetupDefaultContextJob>, IDisposable
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get();

        [NotNull]
        private readonly PlatformAPI _platform;
        [NotNull]
        private readonly ContextApplicator _contextApplicator;
        [NotNull]
        private readonly PooledCompositeDisposable _disposables = new();

        public SetupDefaultContextWorker(
            [NotNull] PlatformAPI platform,
            [NotNull] ContextApplicator contextApplicator)
        {
            _platform = platform;
            _contextApplicator = contextApplicator;
        }

        void IDisposable.Dispose()
        {
            _disposables.Release();
        }

        public void DoJob([NotNull] SetupDefaultContextJob job)
        {
            SetupDefaultContext(job.ConfigPath);
        }

        private void SetupDefaultContext([NotNull] string configPath)
        {
            var path = Path.Combine(Application.streamingAssetsPath, configPath);
            _logger.Log($"{nameof(SetupDefaultContext)} from file={configPath}");

            string json = default;
            if (_platform.RuntimePlatform == RuntimePlatform.Android)
            {
                var request = UnityWebRequest.Get(path);
                if (request is null)
                {
                    _logger.LogError($"Could not setup the default config: failed to get access to {path}");
                }
                else
                {
                    var _ = request.SendWebRequest();
                    while (!request.isDone && !_disposables.IsDisposed)
                    {
                    }
                    if (!_disposables.IsDisposed)
                    {
                        json = request.downloadHandler?.text;
                    }
                }
            }
            else if (File.Exists(path))
            {
                json = File.ReadAllText(path);
            }

            if (string.IsNullOrEmpty(json))
            {
                _logger.LogWarning($"Default config file was not found: {path}");
                _contextApplicator.HandleContext(null, ContextKind.Default);
                return;
            }

            UniTask.RunOnThreadPool(loadContext).Forget();
            return;

            void loadContext()
            {
                var defaultContext = CampaignsContext.Deserialize(json);
                if (defaultContext == null)
                {
                    _logger.LogError($"{nameof(SetupDefaultContext)} from file={configPath} failed. Default context is empty.");
                }
                else
                {
                    _logger.Log($"{nameof(SetupDefaultContext)} from file={configPath} succeeded.");
                }

                _contextApplicator.HandleContext(defaultContext, ContextKind.Default);
            }
        }
    }
}