﻿using System;
using System.Collections.Generic;
using System.Linq;
using JetBrains.Annotations;
using Magify.Model;
using Magify.Rx;
using UnityEngine;

namespace Magify
{
    internal class SubsystemsCollection : ICancelable
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get();
        [NotNull, ItemNotNull]
        private readonly List<object> _subsystems = new();

        public bool IsDisposed { get; private set; }

        public void Register(object instance)
        {
            ThrowIfDisposed();
            _subsystems.Add(instance);
        }

        [NotNull]
        public T Register<T>([NotNull] T instance)
        {
            Register((object)instance);
            return instance;
        }

        public T Register<T>([NotNull] Func<T> creator)
        {
            return Register(creator());
        }

        public void Clear()
        {
            ThrowIfDisposed();
            foreach (var subsystem in _subsystems)
            {
                DisposeSubsystem(subsystem);
            }
            _subsystems.Clear();
        }

        public bool Remove(object subsystem)
        {
            ThrowIfDisposed();
            if (_subsystems.Remove(subsystem))
            {
                DisposeSubsystem(subsystem);
                return true;
            }
            return false;
        }

        public int RemoveAll<T>()
        {
            ThrowIfDisposed();
            var removed = 0;
            for (var i = 0; i < _subsystems.Count;)
            {
                var subsystem = _subsystems[i];
                if (subsystem is T typed)
                {
                    DisposeSubsystem(subsystem);
                    _subsystems.RemoveAt(i);
                    removed++;
                }
                else
                {
                    i++;
                }
            }
            return removed;
        }

        public IEnumerable<T> OfType<T>()
        {
            ThrowIfDisposed();
            return _subsystems.OfType<T>();
        }

        public void ForEach<T>([NotNull] Action<T> action)
        {
            ThrowIfDisposed();
            _subsystems.OfType<T>().ForEach(action);
        }

        public void PreInitializeAll()
        {
            _logger.Log($"{nameof(PreInitializeAll)} called for all subsystems");
            foreach (var subsystem in _subsystems)
                if (subsystem is IPreInitializable preInitializable)
                    preInitializable.PreInitialize();
        }

        public void InitializeAll()
        {
            _logger.Log($"{nameof(InitializeAll)} called for all subsystems");
            foreach (var subsystem in _subsystems)
                if (subsystem is IInitializable initializable)
                    initializable.Initialize();
        }

        public void MigrateAll(MigrationData data)
        {
            _logger.Log($"{nameof(MigrateAll)} called for all subsystems");
            foreach (var subsystem in _subsystems)
                if (subsystem is ILegacyMigrator migrator)
                    migrator.Migrate(data);
        }

        public void UpdateContextForAll([NotNull] CampaignsContext context, ContextKind kind)
        {
            _logger.Log($"{nameof(UpdateContextForAll)} called for {kind} context");
            var noFlags = context.RequestedScopes == ConfigScope.None;
            foreach (var subsystem in _subsystems)
            {
                if (subsystem is not IContextListener contextListener)
                    continue;
                if (kind == ContextKind.Default || noFlags || context.RequestedScopes.HasFlag(contextListener.SuitableScope))
                {
                    _logger.Log($"Context of kind {kind} pushed to the {contextListener.GetType().Name} (scopes: {contextListener.SuitableScope})");
                    contextListener.UpdateContext(context, kind);
                }
            }
        }

        void IDisposable.Dispose()
        {
            if (IsDisposed)
            {
                return;
            }
            Clear();
            IsDisposed = true;
        }

        private void DisposeSubsystem(object subsystem)
        {
            switch (subsystem)
            {
                case ICancelable { IsDisposed: false } cancelable:
                    cancelable.Dispose();
                    break;
                case IDisposable disposable:
                    disposable.Dispose();
                    break;
            }
        }

        private void ThrowIfDisposed()
        {
            if (IsDisposed)
            {
                throw new ObjectDisposedException(nameof(SubsystemsCollection));
            }
        }
    }

    internal static class SubsystemsCollectionExtensions
    {
        [NotNull]
        public static T AddTo<T>([NotNull] this T instance, [NotNull] SubsystemsCollection subsystemsCollection)
        {
            return subsystemsCollection.Register(instance);
        }
    }
}