using System.IO;
using JetBrains.Annotations;
using Newtonsoft.Json;

namespace Magify
{
    internal class FileStorage
    {
        private const string DefaultFolderName = "files";

        [NotNull]
        public string RootFolderPath { get; }

        public FileStorage([NotNull] string storagePath, [NotNull] string folderName = DefaultFolderName)
        {
            RootFolderPath = Path.Combine(storagePath, folderName);
        }

        /// <inheritdoc cref="MagifyDocs.NoThreadSync"/>
        public void Save([NotNull] string fileName, string data)
        {
            File.WriteAllText(GetFilePath(fileName, true), data);
        }

        /// <inheritdoc cref="MagifyDocs.NoThreadSync"/>
        public void SaveJson<T>([NotNull] string fileName, [CanBeNull] T data)
        {
            var filePath = GetFilePath(fileName, true);
            using var stream = File.OpenWrite(filePath);
            using var streamWriter = new StreamWriter(stream);
            using var jsonWriter = new JsonTextWriter(streamWriter);
            JsonFacade.DefaultSerializer.Serialize(jsonWriter, data);
        }

        /// <inheritdoc cref="MagifyDocs.NoThreadSync"/>
        public string Load(string fileName)
        {
            var filePath = GetFilePath(fileName, false);
            return !File.Exists(filePath) ? string.Empty : File.ReadAllText(filePath);
        }

        /// <inheritdoc cref="MagifyDocs.NoThreadSync"/>
        [CanBeNull]
        public T LoadJson<T>([NotNull] string fileName)
        {
            var filePath = GetFilePath(fileName, false);
            if (!File.Exists(filePath))
            {
                return default;
            }

            using var stream = File.OpenRead(filePath);
            using var streamReader = new StreamReader(stream);
            using var jsonReader = new JsonTextReader(streamReader);
            return JsonFacade.DefaultSerializer.Deserialize<T>(jsonReader);
        }

        /// <inheritdoc cref="MagifyDocs.NoThreadSync"/>
        public void Clear(string fileName)
        {
            var filePath = GetFilePath(fileName, false);
            if (Exists(filePath))
            {
                File.Delete(filePath);
            }
        }

        /// <inheritdoc cref="MagifyDocs.NoThreadSync"/>
        /// <param name="sourceFilePath">Must exist</param>
        public void CopyTextFrom(string sourceFilePath, string destinationFileName)
        {
            var destinationFilePath = GetFilePath(destinationFileName, true);
            File.Copy(sourceFilePath, destinationFilePath, true);
        }

        internal bool Exists(string fileName)
        {
            return File.Exists(GetFilePath(fileName, false));
        }

        [NotNull]
        internal string GetFilePath([NotNull] string fileName, bool createDirectory)
        {
            if (createDirectory)
            {
                Directory.CreateDirectory(RootFolderPath);
            }
            return Path.Combine(RootFolderPath, fileName);
        }
    }
}