using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using JetBrains.Annotations;
using Magify.Model;
using Magify.Rx;
using UnityEngine;
using UnityEngine.Pool;

namespace Magify
{
    internal class StoredAppFeaturesProvider : IContextListener, IStoredAppFeaturesCollection, IDisposable
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Features);
        [NotNull]
        private readonly Subject<Unit> _onChanged = new();
        [NotNull]
        private readonly FeaturesBuilder _featuresBuilder;
        [NotNull]
        private readonly PooledCompositeDisposable _disposable = new();
        [NotNull]
        private readonly object _lock = new();
        private int _currentVersion = 0;
        [CanBeNull]
        private IReadOnlyDictionary<string, FeatureConfig> _defaultFeatures = DictionaryUtils<string, FeatureConfig>.Default;
        [CanBeNull]
        private IReadOnlyDictionary<string, FeatureConfig> _currentFeatures = DictionaryUtils<string, FeatureConfig>.Default;

        [NotNull]
        internal IReadOnlyDictionary<string, string> DefaultValues { get; private set; } = DictionaryUtils<string, string>.Default;
        [NotNull]
        internal IReadOnlyDictionary<string, string> CurrentValues { get; private set; } = DictionaryUtils<string, string>.Default;

        [NotNull]
        public IObservable<Unit> OnChanged => _onChanged.ObserveOnMainThread(_disposable.GetOrCreateToken())!;

        public ConfigScope SuitableScope => ConfigScope.StoredAppFeatures;

        public StoredAppFeaturesProvider([NotNull] GeneralPrefs generalPrefs)
        {
            _featuresBuilder = new FeaturesBuilder(generalPrefs);
        }

        public void Dispose()
        {
            lock (_lock)
            {
                _onChanged.Dispose();
                _disposable.Release();
            }
        }

        public void UpdateContext([NotNull] CampaignsContext context, ContextKind kind)
        {
            lock (_lock)
            {
                var features = context.StoredAppFeatures ?? DictionaryUtils<string, FeatureConfig>.Default;
                var parsedFeatures = _featuresBuilder.BuildFeatureValueMap(features).ToDictionary(c => c.Key, c => c.Value as string);
                switch (kind)
                {
                    case ContextKind.Default:
                        _defaultFeatures = features;
                        DefaultValues = parsedFeatures;
                        _onChanged.OnNext(Unit.Default);
                        break;
                    case ContextKind.Downloaded or ContextKind.Saved:
                        _currentFeatures = features;
                        CurrentValues = parsedFeatures;
                        _currentVersion++;
                        _onChanged.OnNext(Unit.Default);
                        break;
                }
            }
        }

        internal void UpdateFeaturesOnTrigger()
        {
            lock (_lock)
            {
                var defaultValues = _featuresBuilder.BuildFeatureValueMap(_defaultFeatures).ToDictionary(c => c.Key, c => c.Value as string);
                var currentValues = _featuresBuilder.BuildFeatureValueMap(_currentFeatures).ToDictionary(c => c.Key, c => c.Value as string);

                var defaultEquals = defaultValues.Values.EqualsInAnyOrder(defaultValues.Count, DefaultValues.Values, DefaultValues.Count);
                var currentEquals = currentValues.Values.EqualsInAnyOrder(currentValues.Count, CurrentValues.Values, CurrentValues.Count);

                if (!defaultEquals || !currentEquals)
                {
                    DefaultValues = defaultValues;
                    CurrentValues = currentValues;
                    _currentVersion++;
                    _onChanged.OnNext(Unit.Default);
                }
            }
        }

        #region IStoredAppFeaturesCollection implementation

        public bool Contains([NotNull] string featureName)
        {
            lock (_lock)
            {
                return CurrentValues.ContainsKey(featureName) || DefaultValues.ContainsKey(featureName);
            }
        }

        public bool TryGet([NotNull] string featureName, out StoredAppFeature feature)
        {
            lock (_lock)
            {
                if (CurrentValues.TryGetValue(featureName, out var url) && url != null)
                {
                    feature = new StoredAppFeature(featureName, url);
                    return true;
                }
                if (DefaultValues.TryGetValue(featureName, out url) && url != null)
                {
                    feature = new StoredAppFeature(featureName, url);
                    return true;
                }
                feature = default;
                return false;
            }
        }

        public IEnumerator<StoredAppFeature> GetEnumerator()
        {
            return IterateAllWithChangesSupport().GetEnumerator();
        }

        IEnumerator IEnumerable.GetEnumerator()
        {
            return GetEnumerator();
        }

        [NotNull]
        private IEnumerable<StoredAppFeature> IterateAllWithChangesSupport()
        {
            int version;
            IReadOnlyDictionary<string, string> currentSnapshot;
            IReadOnlyDictionary<string, string> defaultSnapshot;

            lock (_lock)
            {
                version = _currentVersion;
                currentSnapshot = CurrentValues;
                defaultSnapshot = DefaultValues;
            }

            var set = GenericPool<HashSet<string>>.Get()!;
            set.Clear();
            foreach (var feature in iterate(currentSnapshot))
                yield return feature;
            foreach (var feature in iterate(defaultSnapshot))
                yield return feature;
            set.Clear();
            GenericPool<HashSet<string>>.Release(set);
            yield break;

            [NotNull]
            IEnumerable<StoredAppFeature> iterate([NotNull] IReadOnlyDictionary<string, string> features)
            {
                foreach (var (feature, url) in features)
                {
                    if (!set.Add(feature))
                    {
                        continue;
                    }
                    if (feature == null)
                    {
                        _logger.LogWarning("Null was found as the name of the stored app feature. Skipping...");
                        continue;
                    }
                    if (url == null)
                    {
                        _logger.LogWarning($"Null was found as the url of the stored app feature by name {feature}. Skipping...");
                        continue;
                    }

                    yield return new StoredAppFeature(feature, url);

                    lock (_lock)
                    {
                        if (version != _currentVersion)
                        {
                            _logger.LogWarning("Current stored app features were changed, active iterator will be stopped.");
                            yield break;
                        }
                    }
                }
            }
        }

        #endregion
    }
}