#if UNITY_EDITOR || UNITY_ANDROID
using System;
using System.IO;
using JetBrains.Annotations;
using Magify.Model;
using Magify.Rx;
using UnityEngine;

namespace Magify
{
    internal partial class PlatformAndroid : PlatformAPI, IInitializable
    {
        internal const string LegacyCountersFileName = "legacy-counters.bin";
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get();

        [NotNull]
        private readonly PooledCompositeDisposable _disposable = new();
        private readonly Subject<object> _onBackground = new();
        private readonly Subject<object> _onForeground = new();
        private readonly Subject<LocaleData> _onLocaleChanged = new();
        private readonly Subject<string> _onNewPackageInstalled = new();
        [NotNull]
        private readonly AndroidJavaObject _magifyClient;
        private readonly AndroidLegacyCountersStorage _legacyCountersStorage;
        [CanBeNull]
        private string _cachedSdkInt;
        [CanBeNull]
        private string _cachedAndroidId;
        [CanBeNull]
        private string _cachedGpsAdId;
        [CanBeNull]
        private string _cachedBuildNumber;
        [CanBeNull]
        private bool? _cachedIsTablet;
        [CanBeNull]
        private float? _cachedScreenDpi;

        public override RuntimePlatform RuntimePlatform => RuntimePlatform.Android;

        /// <remarks>
        /// Important: switching to main thread may cause foreground/background events to be skipped when enter background/overlapping with native UI.
        /// </remarks>
        public override IObservable<object> OnBackground => _onBackground;
        public override IObservable<object> OnForeground => _onForeground;
        public override IObservable<LocaleData> OnLocaleChanged => _onLocaleChanged.ObserveOnMainThread(_disposable.GetOrCreateToken());
        public override IObservable<string> OnNewPackageInstalled => _onNewPackageInstalled.ObserveOnMainThread(_disposable.GetOrCreateToken());

        public const int DensityMedium = 160;
        public const int DensityHigh = 240;
        public const int DensityXhigh = 320;
        public const int DensityXxhigh = 480;
        public const int DensityXxxhigh = 640;

        public override string DeviceName => $"{DeviceType}-{Density}";
        public override string DeviceVersion => GetSDKInt();
        public override string BuildNumber
        {
            get
            {
                return
#if MAGIFY_VERBOSE_LOGGING || MAGIFY_DEBUG || UNITY_EDITOR
                    CustomBuildNumber ??
#endif
                    _cachedBuildNumber ?? _magifyClient.Call<long>("getBuildNumber").ToString();
            }
        }

        private string DeviceType => _cachedIsTablet ?? _magifyClient.Call<bool>("isTablet") ? "tablet" : "phone";

        private string Density => (_cachedScreenDpi ?? Screen.dpi) switch
        {
            >= DensityXxxhigh => "xxxhdpi",
            >= DensityXxhigh => "xxhdpi",
            >= DensityXhigh => "xhdpi",
            >= DensityHigh => "hdpi",
            >= DensityMedium => "mdpi",
            _ => "ldpi",
        };

        public PlatformAndroid([NotNull] string storagePath)
        {
            _legacyCountersStorage = AndroidLegacyCountersStorage.Create(Path.Combine(storagePath, LegacyCountersFileName)).AddTo(_disposable);

            using var clsClient = new AndroidJavaClass("com.magify.androidsdk.MagifyClient");
            _magifyClient = clsClient.CallStatic<AndroidJavaObject>("getInstance").AddTo(_disposable);

            using var unityPlayer = new AndroidJavaClass("com.unity3d.player.UnityPlayer");
            using var currentActivity = unityPlayer.GetStatic<AndroidJavaObject>("currentActivity");
            using var androidApp = currentActivity!.Call<AndroidJavaObject>("getApplication");
            _magifyClient.Call("setup", androidApp);
        }

        void IInitializable.Initialize()
        {
            _magifyClient.Call("setForegroundStateChangeCallback", new ForegroundStateChangedCallback(_onForeground, _onBackground));
            _magifyClient.Call("setLocaleChangeCallback", new LocaleChangedCallback(_onLocaleChanged));
            _magifyClient.Call("setNewPackageInstallCallback", new NewPackageInstalledCallback(_onNewPackageInstalled));
            _cachedSdkInt = GetSDKInt();
            _cachedAndroidId = _magifyClient.Call<string>("getAndroidId");
            _cachedGpsAdId = _magifyClient.Call<string>("getGpsAdId");
            _cachedBuildNumber = _magifyClient.Call<long>("getBuildNumber").ToString();
            _cachedIsTablet = _magifyClient.Call<bool>("isTablet");
            _cachedScreenDpi = Screen.dpi;
        }

        public override void AddPlatformSpecificData(AuthTokenPayload authToken, GeneralPrefs prefs, bool isGeoIpEnabled, bool syncStateEnabled)
        {
            base.AddPlatformSpecificData(authToken, prefs, isGeoIpEnabled, syncStateEnabled);
            // Pass null if isGeoIpEnabled == true (absent parameter will be treated by server as true)
            // In order to hide this parameter for sake of anti-fraud
            authToken.IsGeoIPEnabled = isGeoIpEnabled ? null : false;
            authToken.GpsAdId = prefs.GpsAdId.Value;
            authToken.AndroidId = syncStateEnabled ? GetAndroidId() : null;
        }

        private string GetSDKInt()
        {
            if (_cachedSdkInt != null)
            {
                return _cachedSdkInt;
            }
            using var version = new AndroidJavaClass("android.os.Build$VERSION");
            return version.GetStatic<int>("SDK_INT").ToString();
        }

        [NotNull]
        public override LocaleData GetCurrentLocale()
        {
            var countryCode = _magifyClient.Call<string>("getCountryCode");
            var language = _magifyClient.Call<string>("getLanguageWithRegion");
            return new LocaleData(countryCode, language);
        }

        public override void Dispose()
        {
            if (IsDisposed)
            {
                return;
            }
            IsDisposed = true;
            _magifyClient.Call("dispose");
            _disposable.Release();
            base.Dispose();
        }

        public override void TweakUserLocale(string languageTag)
        {
            _magifyClient.Call("tweakUserLocale", languageTag);
        }

        public override void TrackPromotedApplication(string bundleId)
        {
            // Do nothing
        }

        public override bool IsApplicationInstalled(string bundleId)
        {
            return _magifyClient.Call<bool>("isPackageInstalled", bundleId);
        }

        public override bool IsApplicationInstalled(PromotedApplication application)
        {
            return IsApplicationInstalled(application?.BundleId);
        }

        public override string GetGpsAdId()
        {
#if MAGIFY_VERBOSE_LOGGING || MAGIFY_DEBUG || UNITY_EDITOR
            if (!string.IsNullOrEmpty(CustomGpsAdId))
            {
                return CustomGpsAdId;
            }
#endif
            try
            {
                return _cachedGpsAdId ?? _magifyClient.Call<string>("getGpsAdId");
            }
            catch (Exception e)
            {
                _logger.LogException(new MagifyFailedToGetGpsAdIdException(e));
                return null;
            }
        }

        public override string GetAndroidId()
        {
#if MAGIFY_VERBOSE_LOGGING || MAGIFY_DEBUG || UNITY_EDITOR
            if (!string.IsNullOrEmpty(CustomAndroidId))
            {
                return CustomAndroidId;
            }
#endif

            try
            {
                return _cachedAndroidId ?? _magifyClient.Call<string>("getAndroidId");
            }
            catch (Exception e)
            {
                Debug.Log($"Get android id  with exception: {e}");
                return null;
            }
        }

        private class LocaleChangedCallback : AndroidJavaProxy
        {
            private readonly Subject<LocaleData> _localeChangedHandler;

            public LocaleChangedCallback(Subject<LocaleData> localeChangedHandler) : base("com.magify.androidsdk.callback.OnLocaleChangeCallback")
            {
                _localeChangedHandler = localeChangedHandler;
            }

            [UsedImplicitly]
            // ReSharper disable once InconsistentNaming
            public void onChange(string countryCode, string languageWithRegion)
            {
                _logger.Log($"Native locale changed callback received with {nameof(countryCode)}: {countryCode}, {nameof(languageWithRegion)}: {languageWithRegion}");
                var localeData = new LocaleData(countryCode, languageWithRegion);
                _localeChangedHandler?.OnNext(localeData);
            }
        }

        private class ForegroundStateChangedCallback : AndroidJavaProxy
        {
            [NotNull]
            private readonly Subject<object> _foregroundHandler;
            [NotNull]
            private readonly Subject<object> _backgroundHandler;

            public ForegroundStateChangedCallback([NotNull] Subject<object> foregroundHandler, [NotNull] Subject<object> backgroundHandler) : base("com.magify.androidsdk.callback.OnForegroundStateChangeCallback")
            {
                _foregroundHandler = foregroundHandler;
                _backgroundHandler = backgroundHandler;
            }

            [UsedImplicitly]
            // ReSharper disable once InconsistentNaming
            public void onChange(bool isForeground)
            {
                if (isForeground)
                {
                    _foregroundHandler.OnNext(this);
                }
                else
                {
                    _backgroundHandler.OnNext(this);
                }
            }
        }

        private class NewPackageInstalledCallback : AndroidJavaProxy
        {
            [NotNull]
            private readonly Subject<string> _packageInstalledHandler;

            public NewPackageInstalledCallback([NotNull] Subject<string> packageInstalledHandler) : base("com.magify.androidsdk.callback.OnNewPackageInstallCallback")
            {
                _packageInstalledHandler = packageInstalledHandler;
            }

            [UsedImplicitly]
            // ReSharper disable once InconsistentNaming
            public void onNewInstall(string packageName)
            {
                _packageInstalledHandler.OnNext(packageName);
            }
        }
    }
}
#endif