#if UNITY_EDITOR || UNITY_ANDROID
using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using JetBrains.Annotations;
using Magify.Model;
using Newtonsoft.Json;
using UnityEngine;
using UnityEngine.Scripting;
using static Magify.CounterKey.Flags;
using EventType = Magify.Model.EventType;

namespace Magify
{
    internal partial class PlatformAndroid
    {
        private const string ActivationCountersPrefix = "activation_";
        private static readonly MagifyLogger _migrationLogger = MagifyLogger.Get(LoggingScope.Migration);

        public override MigrationData LoadLegacyNativeDataForMigration()
        {
            _magifyClient.Call("startLegacyDataMigration");
            MigrateLegacyCounters();
            return CreateMigrationData();
        }

        public override void CleanLegacyNativeDataAfterMigration()
        {
            _magifyClient.Call("cleanLegacyDataAfterMigration");
        }

        public override int? GetGlobalCounterFromLegacyData(CounterType type, CounterKey key)
        {
            if (_legacyCountersStorage.Counters == null || _legacyCountersStorage.Counters.Count == 0)
            {
                return null;
            }

            var counterKey = key.GetUsedProperties() switch
            {
                CampaignNameProp => createKey($"{key.CampaignName}"),
                CampaignTypeProp => createKey($"{key.CampaignType?.ToEnumString()}"),
                SourceProp => createKey($"{key.Source}"),
                SourceTypeProp => createKey($"{key.SourceType?.ToEnumString()}"),
                NestedNameProp => createKey($"{key.NestedName}"),
                NestedTypeProp => createKey($"{key.NestedType}"),
                CampaignNameProp | SourceProp => createKey($"{key.CampaignName}-{key.Source}"),
                CampaignNameProp | SourceTypeProp => createKey($"{key.CampaignName}-{key.SourceType?.ToEnumString()}"),
                CampaignNameProp | SourceProp | SourceTypeProp => createKey($"{key.CampaignName}-{key.Source}-{key.SourceType?.ToEnumString()}"),
                CampaignTypeProp | SourceTypeProp => createKey($"{key.CampaignType?.ToEnumString()}-{key.SourceType?.ToEnumString()}"),
                _ => null
            };

            _migrationLogger.Log($"Looking for legacy counter key: {counterKey}");
            if (counterKey == null || !_legacyCountersStorage.Counters.TryGetValue(counterKey, out var counterValue))
            {
                return null;
            }

            _migrationLogger.Log($"Got legacy counter value: {counterKey}={counterValue}");
            _legacyCountersStorage.Counters.Remove(counterKey);
            return counterValue;

            string createKey(string entity)
            {
                // LtoStart counters stored in separated lto.prefs file. And they have different format.
                if (type is CounterType.LtoStart)
                {
                    return $"{ActivationCountersPrefix}counter.{entity}";
                }

                var typeSuffix = type switch
                {
                    CounterType.Bonuses => "global_bonuses",
                    CounterType.Clicks => "global_clicks",
                    CounterType.Events => "global_events",
                    CounterType.Impressions => "global_impressions",
                    CounterType.Rewards => "global_rewards",
                    CounterType.Nested => "global_nested",
                    _ => ""
                };
                return $"{entity}.{typeSuffix}";
            }
        }

        private void MigrateLegacyCounters()
        {
            var fileContent = default(string);
            // Global counters migration
            try
            {
                var countersFilePath = _magifyClient.Call<string>("getLegacyCountersFile");
                if (File.Exists(countersFilePath))
                {
                    fileContent = File.ReadAllText(countersFilePath);
                    var counters = JsonFacade.DeserializeObject<Dictionary<string, int>>(fileContent);
                    if (counters != null)
                    {
                        if (_legacyCountersStorage.Counters.Count == 0)
                        {
                            _legacyCountersStorage.Counters.AddRange(counters);
                        }
                    }
                    _migrationLogger.Log($"Global counters: ({countersFilePath}) \n{fileContent}");
                }
                else
                {
                    _migrationLogger.Log("There is no old Global counters file");
                }
            }
            catch (Exception e)
            {
                var exception = new MagifyMigrationFailedException("Global counters", fileContent, e);
                _migrationLogger.LogError(exception.Message);
            }

            // Activation counters migration
            try
            {
                var ltoPrefsFilePath = _magifyClient.Call<string>("getLegacyLtoPrefsFile");
                if (File.Exists(ltoPrefsFilePath))
                {
                    fileContent = File.ReadAllText(ltoPrefsFilePath);
                    var counters = JsonFacade.DeserializeObject<Dictionary<string, int>>(fileContent);
                    if (counters != null)
                    {
                        foreach (var (originalKey, value) in counters)
                        {
                            var key = ActivationCountersPrefix + originalKey;
                            if (!_legacyCountersStorage.Counters.TryAdd(key, value))
                            {
                                _migrationLogger.LogError($"Failed to add activation counter ({key}:{value}), already exist: {key}:{_legacyCountersStorage.Counters[key]}");
                            }
                        }
                    }
                    _migrationLogger.Log($"Activation counters: ({ltoPrefsFilePath}) \n{fileContent}");
                }
                else
                {
                    _migrationLogger.Log("There is no old Activation counters file");
                }
            }
            catch (Exception e)
            {
                var exception = new MagifyMigrationFailedException("Activation counters", fileContent, e);
                _migrationLogger.LogError(exception.Message);
            }
        }

        private MigrationData CreateMigrationData()
        {
            var migrationData = new MigrationData
            {
                ContextPath = _magifyClient.Call<string>("getLegacyContextFile")
            };

            PrepareMigratedPrefs(migrationData);
            PrepareMigratedAnalytics(migrationData);
            PrepareMigratedLtoModels(migrationData);
            PrepareMigratedPurchaseRecords(migrationData);

            return migrationData;
        }

        private void PrepareMigratedPrefs([NotNull] MigrationData migrationData)
        {
            var fileContent = default(string);
            try
            {
                var prefsFilePath = _magifyClient.Call<string>("getLegacyPrefsFile");
                if (!File.Exists(prefsFilePath))
                {
                    return;
                }

                fileContent = File.ReadAllText(prefsFilePath);
                var prefs = JsonFacade.DeserializeObject<Dictionary<string, object>>(fileContent);
                if (prefs == null)
                {
                    return;
                }

                migrationData.ClientId = prefs.TryGetValue("client_id", out var clientId) ? clientId as string : null;
                migrationData.FirstInstalledVersion = prefs.TryGetValue("first_version", out var firstVersion) ? firstVersion as string : null;
                var firstLaunchMillis = GetNullable<long>(prefs, "first_launch");
                if (firstLaunchMillis != null)
                {
                    migrationData.FirstLaunchDate = DateTime.UnixEpoch + TimeSpan.FromMilliseconds(firstLaunchMillis.Value);
                }

                migrationData.SubscriptionStatus = (prefs.GetValueOrDefault("subscription_state") as string)?.ToEnumMember<SubscriptionStatus>();
                migrationData.InAppStatus = (prefs.GetValueOrDefault("in_app_status") as string)?.ToEnumMember<InAppStatus>();
                migrationData.AuthorizationStatus = (prefs.GetValueOrDefault("authorization_status") as string)?.ToEnumMember<AuthorizationStatus>();
                migrationData.ReferrerId = prefs.TryGetValue("referrer", out var referrerId) ? referrerId as string : null;

                migrationData.AdjustId = prefs.TryGetValue("adjust_id", out var adjustId) ? adjustId as string : null;
                migrationData.GpsAdId = prefs.TryGetValue("ads_id", out var adsId) ? adsId as string : null;
                migrationData.IsGdprGranted = GetNullable<bool>(prefs, "gdpr_granted");
                migrationData.UsedDefaultCampaignTypes = GetSet<string>(prefs, "used_default_campaigns");
                var productsPerDateJson = prefs.TryGetValue("purch_products_with_date", out var productsPerDate) ? productsPerDate as string : null;
                if (!string.IsNullOrEmpty(productsPerDateJson))
                {

                    migrationData.ProductsReceiptTimestamps = JsonFacade.DeserializeObject<Dictionary<string, long>>(productsPerDateJson)
                        .ToDictionary(kvp => kvp.Key, kvp => DateTime.UnixEpoch + TimeSpan.FromMilliseconds(kvp.Value));
                }
                migrationData.PurchasedInAppProducts = GetSet<string>(prefs, "purch_products");
                migrationData.PurchasedSubscriptionProducts = GetSet<string>(prefs, "purch_subs_products");
                migrationData.UsedFreeProducts = GetSet<string>(prefs, "free_products");
                migrationData.FirebaseInstanceId = prefs.TryGetValue("app_instance_id", out var firebaseInstanceId) ? firebaseInstanceId as string : null;
                migrationData.GlobalSessionCounter = GetNullable<int>(prefs, "session_counter");
                migrationData.SessionId = prefs.TryGetValue("session_id", out var sessionId) ? sessionId as string : null;
                migrationData.SessionStarted = GetNullable<long>(prefs, "session_open_time");
                migrationData.LastActivity = GetNullable<long>(prefs, "session_last_time");
                migrationData.MediaSource = new MediaSource(
                    networkName: prefs.TryGetValue("source_network_name", out var sourceNetwork) ? sourceNetwork as string : null,
                    campaignName: prefs.TryGetValue("source_campaign_name", out var sourceCampaign) ? sourceCampaign as string : null,
                    groupName: prefs.TryGetValue("source_ad_group", out var sourceAdGroup) ? sourceAdGroup as string : null
                );
            }
            catch (Exception e)
            {
                var exception = new MagifyMigrationFailedException("GeneralPrefs", fileContent, e);
                _migrationLogger.LogError(exception.Message);
            }
        }

        private static ISet<T> GetSet<T>(Dictionary<string, object> dictionary, string key)
        {
            if (dictionary.TryGetValue(key, out var value))
            {
                return value switch
                {
                    ISet<T> set => set,
                    IEnumerable<T> enumerable => new HashSet<T>(enumerable),
                    IEnumerable enumerable => new HashSet<T>(enumerable.Cast<T>()),
                    _ => throw new MagifyMigrationWrongTypeException(typeof(ISet<T>), value.GetType()),
                };
            }
            return null;
        }

        private static T? GetNullable<T>(Dictionary<string, object> dictionary, string key)
            where T: struct
        {
            if (dictionary.TryGetValue(key, out var value))
            {
                if (value is T casted)
                {
                    return casted;
                }
                try
                {
                    return (T)Convert.ChangeType(value, typeof(T));
                }
                catch (Exception e)
                {
                    throw new MagifyMigrationWrongTypeException(typeof(T), value.GetType(), e);
                }
            }
            return null;
        }

        private void PrepareMigratedAnalytics([NotNull] MigrationData migrationData)
        {
            migrationData.AppLaunchEvents = loadEvents<AppLaunchEvent>(EventType.AppLaunch);
            migrationData.CustomSessionEvents = loadEvents<CustomSessionEvent>(EventType.SessionEvent);
            migrationData.AppBackgroundEvents = loadEvents<AppBackgroundEvent>(EventType.AppBackgroundEvent);
            migrationData.TransactionEvents = loadEvents<TransactionEvent>(EventType.Transaction);
            migrationData.FirebaseMappingEvents = loadEvents<FirebaseMappingEvent>(EventType.FirebaseMapping);
            migrationData.MailingStatusMappingEvents = loadEvents<MailingStatusMappingEvent>(EventType.MailingStatusMapping);
            migrationData.ImpressionEvents = loadEvents<CampaignImpressionEvent>(EventType.Impression);
            migrationData.ApplovinAdsImpressionEvents = loadEvents<AdsImpressionEvent>(EventType.ApplovinAdsImpression);
            migrationData.IronSourceAdsImpressionEvents = loadEvents<AdsImpressionEvent>(EventType.IronSourceAdsImpression);
            migrationData.ImpressionFailEvents = loadEvents<ImpressionFailEvent>(EventType.ImpressionFail);
            migrationData.TrialActivationEvents = loadEvents<ProductPurchaseEvent>(EventType.TrialActivation);
            migrationData.PaidSubscriptionActivationEvents = loadEvents<ProductPurchaseEvent>(EventType.PaidSubscriptionActivation);
            migrationData.InAppEvents = loadEvents<ProductPurchaseEvent>(EventType.InApp);
            migrationData.ClickEvents = loadEvents<CampaignImpressionEvent>(EventType.Click);
            migrationData.AdsClickEvents = loadEvents<CampaignImpressionEvent>(EventType.AdsClick);
            return;

            [CanBeNull]
            IReadOnlyCollection<T> loadEvents<T>(EventType eventType) where T : class, IAnalyticsEvent
            {
                try
                {
                    var eventsFilePath = _magifyClient.Call<string>("getLegacyAnalyticsFile", eventType.ToEnumString());
                    if (!File.Exists(eventsFilePath))
                    {
                        return default;
                    }

                    var serializedEvents = File.ReadAllText(eventsFilePath);
                    return DeserializeAndTruncateIfNeed<List<T>>(serializedEvents);
                }
                catch (Exception e)
                {
                    var exception = new MagifyMigrationFailedException($"[Analytics] {eventType}", string.Empty, e);
                    _migrationLogger.LogError(exception.Message);
                }
                return default;
            }
        }

        private void PrepareMigratedLtoModels([NotNull] MigrationData migrationData)
        {
            var fileContent = default(string);
            try
            {
                var ltoFilePath = _magifyClient.Call<string>("getLegacyLtoFile");
                if (!File.Exists(ltoFilePath))
                {
                    return;
                }

                fileContent = File.ReadAllText(ltoFilePath);
                var ltoModelsData = DeserializeAndTruncateIfNeed<LtoMigrationData>(fileContent);
                migrationData.ActiveTimeLimitedOffers = ltoModelsData?.LtoModels.Select(mapLtoModel).ToList();

                LtoModel mapLtoModel(LtoMigrationModel legacyModel)
                {
                    return new LtoModel
                    {
                        CampaignName = legacyModel.CampaignName,
                        Spot = legacyModel.Spot,
                        StartTime = DateTime.UnixEpoch + TimeSpan.FromMilliseconds(legacyModel.StartTimeMillis),
                        EndTime = DateTime.UnixEpoch + TimeSpan.FromMilliseconds(legacyModel.EndTimeMillis),
                        HasPriority = legacyModel.HasPriority ?? false,
                        Trigger = legacyModel.Trigger,
                        BadgePlaceholder = legacyModel.BadgePlaceholder
                    };
                }
            }
            catch (Exception e)
            {
                var exception = new MagifyMigrationFailedException("Lto models", fileContent, e);
                _migrationLogger.LogError(exception.Message);
            }
        }

        private void PrepareMigratedPurchaseRecords([NotNull] MigrationData migrationData)
        {
            try
            {
                var purchasesFilePath = _magifyClient.Call<string>("getLegacyPurchasesFile");
                if (!File.Exists(purchasesFilePath))
                {
                    return;
                }

                var serializedPurchasesData = File.ReadAllText(purchasesFilePath);
                var records = DeserializeAndTruncateIfNeed<List<PurchaseRecord>>(serializedPurchasesData);
                migrationData.PurchasedRecords = records;
            }
            catch (Exception e)
            {
                var exception = new MagifyMigrationFailedException("Lto models", "", e);
                _migrationLogger.LogError(exception.Message);
            }
        }

        [CanBeNull]
        private static T DeserializeAndTruncateIfNeed<T>([NotNull] string json)
            where T: class
        {
            try
            {
                return JsonFacade.DeserializeObject<T>(json);
            }
            catch (JsonReaderException e)
            {
                string truncated = default;
                if (e.LinePosition >= 0 && e.LinePosition < json.Length)
                {
                    truncated = json.Remove(e.LinePosition);
                }
                return truncated is null
                    ? default(T)
                    : JsonFacade.DeserializeObject<T>(truncated);
            }
        }

        [Preserve]
        internal class LtoMigrationData
        {
            [Preserve]
            [JsonProperty("state")]
            public List<LtoMigrationModel> LtoModels { get; init; }
        }

        [Preserve]
        internal class LtoMigrationModel
        {
            [Preserve]
            [JsonProperty("campaign")]
            public string CampaignName { get; init; }

            [Preserve]
            [JsonProperty("spot")]
            public string Spot { get; init; }

            [Preserve]
            [JsonProperty("startTime")]
            public long StartTimeMillis { get; init; }

            [Preserve]
            [JsonProperty("endTime")]
            public long EndTimeMillis { get; init; }

            [Preserve]
            [JsonProperty("hasPriority")]
            public bool? HasPriority { get; init; }

            [Preserve]
            [CanBeNull]
            [JsonProperty("launchedTrigger")]
            public string Trigger { get; init; }

            [Preserve]
            [CanBeNull]
            [JsonProperty("spotImage")]
            public string BadgePlaceholder { get; init; }
        }
    }
}
#endif