using System;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Rx;

namespace Magify
{
    /// <summary>
    /// Thread-safe while <see cref="SingleModeRequestTools"/> and <see cref="IndependentModeRequestTools"/> are thread-safe
    /// </summary>
    internal class RequestContext<TOut> : ICancelable
    {
        [NotNull]
        private readonly object _lock = new();

        [NotNull]
        private CancellationTokenSource _allRequestsToken = new();

        [NotNull]
        private CancellationTokenSource _activeRequestCancellationToken = new();

        [CanBeNull]
        public SingleModeRequestTools<TOut> SingleModeRequestTools { get; }

        [CanBeNull]
        public IndependentModeRequestTools IndependentModeRequestTools { get; }

        [NotNull]
        public ResponseProcessorDelegate<TOut> ResponseProcessor { get; }

        public bool IsDisposed { get; private set; }

        public RequestContext(
            [CanBeNull] SingleModeRequestTools<TOut> singleModeRequestTools,
            [CanBeNull] IndependentModeRequestTools independentModeRequestTools,
            [NotNull] ResponseProcessorDelegate<TOut> responseProcessor)
        {
            SingleModeRequestTools = singleModeRequestTools;
            IndependentModeRequestTools = independentModeRequestTools;
            ResponseProcessor = responseProcessor;
        }

        public void Dispose()
        {
            lock (_lock)
            {
                ThreadingUtils.CancelAndReplace(ref _allRequestsToken, ref _activeRequestCancellationToken);
                SingleModeRequestTools?.Dispose();
                IsDisposed = true;
            }
        }

        public void CancelAllRequests()
        {
            lock (_lock)
            {
                ThreadingUtils.CancelAndReplace(ref _allRequestsToken);
            }
        }

        public void CancelActiveRequest()
        {
            lock (_lock)
            {
                ThreadingUtils.CancelAndReplace(ref _activeRequestCancellationToken);
            }
        }

        public CancellationToken GetAllRequestsToken()
        {
            lock (_lock)
            {
                return _allRequestsToken.Token;
            }
        }

        public CancellationToken GetActiveRequestCancellationToken()
        {
            lock (_lock)
            {
                return _activeRequestCancellationToken.Token;
            }
        }

        public bool CheckSingleRequestInProgress()
        {
            lock (_lock)
            {
                return SingleModeRequestTools?.IsThereActiveCompletionTask() is true;
            }
        }
    }

    internal class SingleModeRequestTools<TOut> : IDisposable
    {
        [NotNull]
        private readonly object _lock = new();

        [CanBeNull]
        private UniTaskCompletionSource<TOut> _completionSource;
        [NotNull]
        private CompositeCancellationTokenSource _singleCancellationTokenSource = new();
        [NotNull]
        public ReusableCancellationTokenSource RequestRepeatingDelayToken { get; } = new();
        [NotNull]
        public ExponentialBackoff Backoff { get; } = new();

        [CanBeNull]
        public UniTaskCompletionSource<TOut> CompletionSourceRaw => _completionSource;

        public bool TryCreateNewCompletionSource(in CancellationToken cancellationToken, [NotNull] out UniTask<TOut>? completionTask, out CancellationToken singleCancellationToken)
        {
            lock (_lock)
            {
                if (_singleCancellationTokenSource.IsCancellationRequested)
                {
                    ThreadingUtils.DiaposeAndReplace(ref _singleCancellationTokenSource);
                }
                _singleCancellationTokenSource.Register(cancellationToken);
                singleCancellationToken = _singleCancellationTokenSource.Token;

                if (_completionSource == null)
                {
                    _completionSource = new UniTaskCompletionSource<TOut>();
                    completionTask = _completionSource.Task!;
                    return true;
                }

                completionTask = _completionSource.Task!;
                return false;
            }
        }

        public bool IsThereActiveCompletionTask()
        {
            lock (_lock)
            {
                return _completionSource?.Task != null && _completionSource?.InProgress() is true;
            }
        }

        public void TrySetResult([CanBeNull] TOut result)
        {
            lock (_lock)
            {
                ThreadingUtils.TrySetResultAndRemove_CancelAndReplaceToken(ref _completionSource, result, ref _singleCancellationTokenSource);
            }
        }

        public void TrySetCancelled(CancellationToken cancellationToken)
        {
            lock (_lock)
            {
                ThreadingUtils.TryCancelAndRemove_CancelAndReplaceToken(ref _completionSource, cancellationToken, ref _singleCancellationTokenSource);
            }
        }

        public void TrySetException([NotNull] Exception exception)
        {
            lock (_lock)
            {
                ThreadingUtils.TryExceptionAndRemove_CancelAndReplaceToken(ref _completionSource, exception, ref _singleCancellationTokenSource);
            }
        }

        public void ResetRequestsRepeatDelay()
        {
            RequestRepeatingDelayToken.CancelCurrent();
        }

        public void Dispose()
        {
            lock (_lock)
            {
                RequestRepeatingDelayToken.CancelCurrent();
                ThreadingUtils.TryCancelAndRemove_CancelAndReplaceToken(ref _completionSource, CancellationToken.None, ref _singleCancellationTokenSource);
            }
        }
    }

    internal class IndependentModeRequestTools
    {
        [NotNull]
        public ExponentialBackoff Backoff { get; }

        [NotNull]
        public ReusableCancellationTokenSource RequestRepeatingDelayToken { get; }

        public IndependentModeRequestTools()
        {
            Backoff = new ExponentialBackoff();
            RequestRepeatingDelayToken = new ReusableCancellationTokenSource();
        }

        public IndependentModeRequestTools(ulong delayMilliseconds, ulong maxDelayMilliseconds)
        {
            Backoff = new ExponentialBackoff(delayMilliseconds, maxDelayMilliseconds);
            RequestRepeatingDelayToken = new ReusableCancellationTokenSource();
        }
    }
}