using System;
using JetBrains.Annotations;

namespace Magify
{
    public readonly struct EndpointUrl : IEquatable<EndpointUrl>
    {
        public static readonly EndpointUrl Production = new("https://api.magify.com");
        public static readonly EndpointUrl Staging = new("https://api.jx-staging.magify.com/");
        public static readonly EndpointUrl Offline = new(string.Empty);
#if MAGIFY_VERBOSE_LOGGING || MAGIFY_DEBUG || UNITY_EDITOR
        public static readonly EndpointUrl Dev = new("https://eagle-jx-development.jx-dev.k8s.magify.com/");
#endif

        [NotNull]
        public readonly string Value;
        public bool IsOffline => string.IsNullOrEmpty(Value);

        public EndpointUrl([NotNull] string value)
        {
            Value = value;
        }

        public static bool operator ==(EndpointUrl a, EndpointUrl b) => a.Value == b.Value;

        public static bool operator !=(EndpointUrl a, EndpointUrl b) => !(a == b);

        public bool Equals([CanBeNull] EndpointUrl other) => Value == other.Value;

        public override bool Equals([CanBeNull] object obj) => obj is EndpointUrl other && Equals(other);

        public override int GetHashCode() => Value.GetHashCode();

        internal static EndpointUrl FromEnvironment(Environment environment)
        {
            return environment switch
            {
                Environment.Production => Production,
                Environment.Staging => Staging,
                Environment.Offline => Offline,
#if MAGIFY_VERBOSE_LOGGING || MAGIFY_DEBUG || UNITY_EDITOR
                Environment.Dev => Dev,
#endif
                _ => throw new InvalidMagifyEnvironmentException(environment),
            };
        }
    }
}