﻿using JetBrains.Annotations;
using UnityEngine;

namespace Magify
{
    internal class AuthorizationToken
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get();
        [NotNull]
        private readonly object _lock = new();
        [NotNull]
        private readonly GeneralPrefs _generalPrefs;

        public bool IsExpired
        {
            get
            {
                lock (_lock)
                {
                    return _generalPrefs.AuthorizationTokenExpired.Value;
                }
            }
            set
            {
                lock (_lock)
                {
                    SetExpiredThreadUnsafe(value);
                }
            }
        }

        public string AuthToken
        {
            get
            {
                lock (_lock)
                {
                    return _generalPrefs.AuthorizationToken.Value;
                }
            }
            set
            {
                lock (_lock)
                {
                    SetExpiredThreadUnsafe(false);
                    _generalPrefs.AuthorizationToken.Value = value;
                }
            }
        }

        public AuthorizationToken([NotNull] GeneralPrefs generalPrefs)
        {
            _generalPrefs = generalPrefs;
        }

        public bool HasValidToken()
        {
            lock (_lock)
            {
                return !_generalPrefs.AuthorizationTokenExpired.Value
                    && !string.IsNullOrEmpty(_generalPrefs.AuthorizationToken.Value);
            }
        }

        /// <inheritdoc cref="MagifyDocs.NoThreadSync"/>
        private void SetExpiredThreadUnsafe(bool value)
        {
            if (value && string.IsNullOrEmpty(_generalPrefs.AuthorizationToken.Value))
            {
                _logger.Log("Skip setting auth token's expired flag to 'True' because token has been reset.");
                _generalPrefs.AuthorizationTokenExpired.Value = false;
            }
            else _generalPrefs.AuthorizationTokenExpired.Value = value;
        }
    }
}