using System;
using System.Collections.Generic;
using System.IO;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Rx;
using Newtonsoft.Json;
using UnityEngine;
using Magify.Features;

namespace Magify
{
    internal partial class MinimalMagifyClient : IMagifyClient
    {
        private static readonly MagifyLogger _logger = MagifyLogger.Get();

        private readonly string _configPath;
        [NotNull]
        private readonly PooledCompositeDisposable _disposables = new();
        private readonly SubsystemsCollection _subsystems = new();
        private readonly GeneralPrefs _generalPrefs;
        [NotNull]
        private readonly AuthorizationConfigBuilder _authBuilder;
        [NotNull]
        private readonly AppStatePrefs _appStatePrefs;
        [NotNull]
        private readonly ClientIdProvider _clientIdProvider;
        private readonly IServerApi _serverApi;
        private readonly MinimalCampaignsTracker _campaignsTracker;
        private readonly MinimalAnalyticsTracker _analyticsTracker;
        private readonly AnalyticsSessionTracker _analyticsSessionTracker;
        private readonly SessionCounter _sessionCounter;
        [NotNull]
        private readonly FeaturesProvider _featuresProvider;
        [NotNull]
        private readonly StoredAppFeaturesProvider _storedAppFeaturesProvider;
        [NotNull]
        private readonly FeaturesUpdater _featuresUpdater;
        [NotNull]
        private readonly ContentProvider _contentProvider;

        private readonly PurchaseTracker _purchaseTracker;
        private readonly MinimalAnalyticsService _analyticsService;
        [NotNull]
        private readonly AppVersionProvider _appVersionProvider;
        [NotNull]
        private readonly PlatformAPI _platform;
        [NotNull]
        private readonly ContextApplicator _contextApplicator;
        [NotNull]
        private readonly ContextFileStorage _contextFileStorage;
        [NotNull]
        private readonly MagifyWorkersManager _workerManager;
        [NotNull]
        private readonly UniTaskCompletionSource _defaultContextParsingCompletionSource = new();
        [NotNull]
        private readonly UniTaskCompletionSource _savedContextParsingCompletionSource = new();

        private bool _isSetup;

        public bool IsDisposed => _disposables.IsDisposed;
        public IFeatures Features => _featuresProvider;
        public string ClientId => _generalPrefs.ClientId.Value;
        public bool CustomClientIdWasSet => _generalPrefs.CustomClientIdWasSet.Value;
        public IReactiveProperty<bool> IsGeoIpEnabled { get; }
        public IReactiveProperty<Environment> Environment { get; }
        [NotNull]
        public IReactiveProperty<IPurchaseVerificationHandler> ExternalPurchaseVerificationHandler => _purchaseTracker.ExternalPurchaseVerificationHandler;
        [NotNull]
        public IReactiveProperty<float> VerificationRetryInterval => _purchaseTracker.VerificationRetryInterval;
        public IReactiveProperty<SubscriptionStatus> SubscriptionStatus => _generalPrefs.SubscriptionStatus;
        public IReactiveProperty<InAppStatus> InAppStatus => _generalPrefs.InAppStatus;
        public IReactiveProperty<string> AdjustId => _generalPrefs.AdjustId;
        public IReactiveProperty<string> FirebaseInstanceId => _generalPrefs.FirebaseInstanceId;
        public IReactiveProperty<bool> IsGdprGranted => _generalPrefs.IsGdprGranted;
        public IReadOnlyReactiveProperty<string> GpsAdId => _generalPrefs.GpsAdId;
        public IReadOnlyReactiveProperty<bool?> AttAuthorizationStatus => _generalPrefs.AttAuthorizationStatus;
        [NotNull]
        public IReactiveProperty<ConfigScope> RemoteContextScopes => _generalPrefs.RemoteContextScopes;

        public bool IsTablet => Platform.IsTablet;
        public bool IsPortrait => Screen.width < Screen.height;
        public ContextSyncTime LastContextSyncTime { get; private set; }
        public IReactiveProperty<TimeSpan> SessionsInterval => _sessionCounter.Interval;
        public IReadOnlyReactiveProperty<int> SessionNumber => _generalPrefs.GlobalSessionCounter;
        public PlatformAPI Platform => _platform;
        public string FirstInstalledVersion => _generalPrefs.FirstInstalledVersion;
        public DateTime FirstLaunchDate => _generalPrefs.FirstLaunchDate;
        [NotNull]
        public IStoredAppFeaturesCollection StoredAppFeaturesProvider => _storedAppFeaturesProvider;

        public event Action<ConfigKind> OnConfigParsed;
        public event Action<ConfigKind> OnConfigParsedOnMainThread;
        public event Action OnConfigLoaded;
        public event Action OnConfigSyncRequested;

        [NotNull]
        public string AppVersion
        {
            get => _appVersionProvider.AppVersion;
            set
            {
                _appVersionProvider.Custom(value);
                _serverApi.ResetAuthorizationToken();
            }
        }

#pragma warning disable CS0067 // The event is never used
        public event Action OnPurchasedProductsChanged;
        public event Action OnUserDidTakeScreenshot;
#pragma warning restore
        public event Action OnApplicationEnterForeground;
        public event Action OnApplicationEnterBackground;

        public MinimalMagifyClient(string appName, [NotNull] string configPath, string rootPath, Environment environment, EditorDevice editorDevice = default)
            : this(appName, configPath, rootPath, customClientId: null, environment, editorDevice, new NetworkClient())
        {
        }

        public MinimalMagifyClient(string appName, [NotNull] string configPath, string rootPath, CustomClientId customClientId, Environment environment, EditorDevice editorDevice = default)
            : this(appName, configPath, rootPath, customClientId, environment, editorDevice, new NetworkClient())
        {
        }

        internal MinimalMagifyClient(string appName, [NotNull] string configPath, string rootPath, Environment environment, EditorDevice editorDevice, INetworkClient networkClient)
            : this(appName, configPath, rootPath, customClientId: null, environment, editorDevice, networkClient)
        {
        }

        internal MinimalMagifyClient(string appName, [NotNull] string configPath, string rootPath, [CanBeNull] CustomClientId customClientId, Environment environment, EditorDevice editorDevice, INetworkClient networkClient)
        {
            _configPath = configPath;
            Environment = new ReactiveProperty<Environment>(environment);

            _subsystems.AddTo(_disposables);
            _appVersionProvider = new AppVersionProvider().AddTo(_subsystems);
            _platform = PlatformAPI.PreparePlatformBridge(editorDevice, rootPath).AddTo(_subsystems);
            _contextFileStorage = new ContextFileStorage(rootPath).AddTo(_subsystems);
            _contextApplicator = new ContextApplicator(_subsystems).AddTo(_subsystems);
            var generalPrefsPath = Path.Combine(rootPath, IMagifyClient.GeneralPrefsName);
            var localGeneralPrefsPath = Path.Combine(rootPath, IMagifyClient.LocalGeneralPrefsName);
            _generalPrefs = GeneralPrefs.Create(generalPrefsPath, localGeneralPrefsPath, _appVersionProvider).AddTo(_subsystems);
            _appStatePrefs = AppStatePrefs.Create(Path.Combine(rootPath, IMagifyClient.AppStatePrefsName)).AddTo(_subsystems);
            _clientIdProvider = new ClientIdProvider(_generalPrefs, _appStatePrefs, _platform, customClientId).AddTo(_subsystems);

            _authBuilder = new AuthorizationConfigBuilder(appName, _appVersionProvider, Platform, _generalPrefs).AddTo(_subsystems);

            _campaignsTracker = new MinimalCampaignsTracker(_generalPrefs).AddTo(_subsystems);

            _serverApi = new ServerApi(EndpointUrl.FromEnvironment(Environment.Value), networkClient, _generalPrefs, _appStatePrefs, _authBuilder, _platform).AddTo(_subsystems);

            _analyticsTracker = new MinimalAnalyticsTracker(_serverApi, _generalPrefs, _appStatePrefs, _platform, rootPath, _appVersionProvider).AddTo(_subsystems);
            _analyticsSessionTracker = new AnalyticsSessionTracker(_analyticsTracker, _generalPrefs, _appStatePrefs, Platform, _appVersionProvider).AddTo(_subsystems);

            _sessionCounter = new SessionCounter(_generalPrefs, Platform).AddTo(_subsystems);

            _featuresProvider = new FeaturesProvider(_generalPrefs, _analyticsTracker).AddTo(_subsystems);
            _storedAppFeaturesProvider = new StoredAppFeaturesProvider(_generalPrefs).AddTo(_subsystems);
            _featuresUpdater = new FeaturesUpdater(_featuresProvider, _storedAppFeaturesProvider, _generalPrefs).AddTo(_subsystems);

            _purchaseTracker = new PurchaseTracker(_serverApi, Platform, rootPath).AddTo(_subsystems);
            _analyticsService = new MinimalAnalyticsService(_analyticsTracker).AddTo(_subsystems);

            _contentProvider = new ContentProvider().AddTo(_subsystems);

            var workers = new IMagifyWorker[]
            {
                new MagifyContextSyncWorker(() => OnConfigSyncRequested?.Invoke(), syncTime => LastContextSyncTime = syncTime, _serverApi, _contextApplicator, _contextFileStorage),
                new SetupSavedContextWorker(_contextFileStorage, _contextApplicator),
                new SetupDefaultContextWorker(_platform, _contextApplicator),
            };
            _workerManager = new MagifyWorkersManager(workers).AddTo(_subsystems);

            IsGeoIpEnabled = _serverApi.IsGeoIpEnabled;
        }

        public void InitializeSdk()
        {
            using(_generalPrefs.MultipleChangeScope())
                _subsystems.PreInitializeAll();
            if (RemoteContextScopes.Value == ConfigScope.None)
            {
                RemoteContextScopes.Value = ConfigScope.Content | ConfigScope.AppFeatures | ConfigScope.StoredAppFeatures;
            }

            _contextApplicator.ContextApplied
                .Subscribe(contextKind =>
                {
                    _logger.Log($"{contextKind} context has been applied");
                    if (contextKind is ContextKind.Default)
                        _defaultContextParsingCompletionSource.TrySetResult();
                    if (contextKind is ContextKind.Saved)
                        _savedContextParsingCompletionSource.TrySetResult();
                    OnConfigParsed?.Invoke(contextKind.ToConfigKind());
                })
                .AddTo(_disposables);
            _contextApplicator.ContextSkipped
                .Subscribe(contextKind =>
                {
                    _logger.Log($"{contextKind} context has been skipped");
                    if (contextKind is ContextKind.Default)
                        _defaultContextParsingCompletionSource.TrySetResult();
                    if (contextKind is ContextKind.Saved)
                        _savedContextParsingCompletionSource.TrySetResult();
                })
                .AddTo(_disposables);
            _contextApplicator.ContextAppliedOnMainThread
                .Subscribe(contextKind => OnConfigParsedOnMainThread?.Invoke(contextKind.ToConfigKind()))
                .AddTo(_disposables);
            _contextApplicator.ContextAppliedOnMainThread
                .Where(kind => kind is ContextKind.Downloaded)
                .Subscribe(_ => OnConfigLoaded?.Invoke())
                .AddTo(_disposables);

            try
            {
                _workerManager.DoJob(new SetupSavedContextJob());
            }
            catch (Exception e) { _logger.LogException(new MagifyFailedToLoadContextFromDiskException(ContextKind.Saved, e)); }

            try
            {
                _workerManager.DoJob(new SetupDefaultContextJob(_configPath));
            }
            catch (Exception e) { _logger.LogException(new MagifyFailedToLoadContextFromDiskException(ContextKind.Default, e)); }

            if (_generalPrefs.VersionSessionCounter == 0)
            {
                PrepareForFirstSessionInVersion();
            }

            Environment
                .SkipLatestValueOnSubscribe()
                .Subscribe(value => _serverApi.ChangeServerUrl(EndpointUrl.FromEnvironment(value)))
                .AddTo(_disposables);
            RemoteContextScopes
                .SkipLatestValueOnSubscribe()
                .Subscribe(_ => _serverApi.CancelAllContextLoadings())
                .AddTo(_disposables);
            Platform.OnLocaleChanged
                .Subscribe(localeData => _serverApi.RefreshLocale(localeData).IfTrue(() => _serverApi.CancelAllContextLoadings().ContinueWith(Sync)))
                .AddTo(_disposables);
            Platform.OnForeground
                .Subscribe(_ =>
                {
                    _subsystems.ForEach<IForegroundListener>(s => s.OnForeground());
                    OnApplicationEnterForeground?.Invoke();
                })
                .AddTo(_disposables);
            Platform.OnBackground
                .Subscribe(_ =>
                {
                    _subsystems.ForEach<IBackgroundListener>(s => s.OnBackground());
                    OnApplicationEnterBackground?.Invoke();
                })
                .AddTo(_disposables);
            _generalPrefs.AttAuthorizationStatus
                .SkipLatestValueOnSubscribe()
                .Subscribe(CancelRequestsAndSync)
                .AddTo(_disposables);
            _generalPrefs.SessionId
                .SkipLatestValueOnSubscribe()
                .Subscribe(Sync)
                .AddTo(_disposables);
            _clientIdProvider.CloudClientIdReplacedLocal
                .Where(replaced => replaced is true)
                .Subscribe(_ => _serverApi.CancelAllServerInteractions().Forget())
                .AddTo(_disposables);

            _subsystems.ForEach<IInitializable>(c => c.Initialize());
            _generalPrefs.LastInitializedVersion = _appVersionProvider.AppVersion;
        }

        public void Setup()
        {
            if (_isSetup)
            {
                _logger.Log("Magify client has already been setup.");
                return;
            }
            _isSetup = true;

            Sync();
        }

        public void Sync<T>(T _) => Sync();
        public void Sync()
        {
            if (!_isSetup)
            {
                _logger.Log("Cannot perform sync before initial setup.");
                return;
            }
            if (IsDisposed)
            {
                _logger.Log("Cannot perform sync after disposal.");
                return;
            }

            _workerManager.DoJob(new MagifyContextSyncJob(RemoteContextScopes.Value, _disposables.GetOrCreateToken()));
        }

        private void CancelRequestsAndSync<T>(T _) => CancelRequestsAndSync();
        private void CancelRequestsAndSync()
        {
            _serverApi.CancelAllServerInteractions().ContinueWith(Sync);
        }

        public UniTask AwaitInitialConfigParsing(CancellationToken cancellationToken)
        {
            if (_defaultContextParsingCompletionSource.IsCompleted() && _savedContextParsingCompletionSource.IsCompleted())
            {
                return UniTask.CompletedTask;
            }
            var task = UniTask.WhenAll(_defaultContextParsingCompletionSource.Task, _savedContextParsingCompletionSource.Task);
            return task.AttachExternalCancellation(cancellationToken);
        }

        public IList<ContentItem> GetContentList(string group, string key, ICollection<string> tags = null)
        {
            return _contentProvider.GetContentList(group, key, tags);
        }

        public ContentItem GetEarliestContent(string group, string key)
        {
            return _contentProvider.GetEarliestContent(group, key);
        }

        public ContentItem GetLatestContent(string group, string key)
        {
            return _contentProvider.GetLatestContent(group, key);
        }

        public void TweakAnalyticsConfig(int eventsGroupSize, int syncTimeInterval)
        {
            _logger.Log($"Tweak analytics config with: {nameof(eventsGroupSize)}={eventsGroupSize}, {nameof(syncTimeInterval)}={syncTimeInterval}");
            _analyticsService.SetupAnalyticsConfig(eventsGroupSize, syncTimeInterval);
        }

        public void ResetAnalyticsConfig()
        {
            _logger.Log("Resetting analytics config.");
            _analyticsService.ResetAnalyticsConfig();
        }

        public UniTask<string> ForceReissueAuthToken(CancellationToken cancellationToken)
        {
            _serverApi.ResetAuthorizationToken();
            return _serverApi.GetAuthorizationTokenUntilSuccessAsync(cancellationToken);
        }

        public void ResetContext(bool cancelIfLoading)
        {
            if (cancelIfLoading) _serverApi.CancelAllContextLoadings();
            _contextFileStorage.ResetContext();
            ResetCachedConfig();
        }

        private void ResetCachedConfig()
        {
            _contentProvider.ResetContentMap();
            _featuresProvider.ResetFeatures();

            _workerManager.DoJob(new SetupDefaultContextJob(_configPath));
        }

        public void RefreshGpsAdId()
        {
            _serverApi.RefreshGpsAdId();
        }

        public void TweakFirstLaunchDate(DateTime date)
        {
            AssertProductionTweak(string.Intern("first launch date"));
            _logger.Log($"Tweak first launch date to: {date.ToString()}");
            _generalPrefs.FirstLaunchDate = date;
            _authBuilder.UpdateConfig();
            _serverApi.ResetAuthorizationToken();
        }

        public void TweakUserLocale([NotNull] string languageTag)
        {
            AssertProductionTweak(string.Intern("user locale"));
            _logger.Log($"Tweak user locale to: {languageTag}");
            Platform.TweakUserLocale(languageTag);
        }

        public void SetAttStatus(bool authorized)
        {
            _logger.Log($"Set ATT status: {nameof(authorized)}={authorized}");
            _generalPrefs.AttAuthorizationStatus.Value = authorized;
        }

        public void SetMediaSource(string networkName, string campaignName, string adGroup)
        {
            _logger.Log($"{nameof(SetMediaSource)} with networkName={networkName}, campaignName={campaignName}, adGroup={adGroup}");
            var mediaSource = new MediaSource(networkName, campaignName, adGroup);
            if (_generalPrefs.MediaSource.Equals(mediaSource))
            {
                return;
            }
            _generalPrefs.MediaSource = mediaSource;
            Sync();
        }

        public void TrackAdsImpression(string campaignName, IAdsImpression adsImpression)
        {
            _logger.Log($"{nameof(TrackAdsImpression)} with impression\n {JsonFacade.SerializeObject(adsImpression, Formatting.Indented)}");
            _analyticsService.TrackAdsImpression(adsImpression);
        }

        public void TrackExternalSubscriptionActivation(bool isTrial, string productId, string price, string currency, string transactionId, string purchaseToken, string originalTransactionId, string receipt, bool needValidation, PurchaseStore store)
        {
            _analyticsService.TrackSubscription(isTrial, productId, price, currency, transactionId, originalTransactionId, store, Option<string>.None);
            _campaignsTracker.TrackSubscriptionPurchase(productId, InAppSourceKind.External);
            _featuresUpdater.OnProductPurchase();
            if (needValidation)
            {
                _purchaseTracker.ValidateSubscription(productId, transactionId, purchaseToken, receipt);
            }
        }

        public void TrackExternalInAppFor(string productId, string price, string currency, string transactionId, string purchaseToken, string originalTransactionId, string receipt, bool needValidation, PurchaseStore store)
        {
            _analyticsService.TrackInApp(productId, price, currency, transactionId, originalTransactionId, store, Option<string>.None);
            _campaignsTracker.TrackInAppPurchase(productId, InAppSourceKind.External);
            _featuresUpdater.OnProductPurchase();
            if (needValidation)
            {
                _purchaseTracker.ValidateInApp(productId, transactionId, purchaseToken, receipt);
            }
        }

        public void TrackTrustedPurchase([NotNull] TrustedPurchaseRecord record, bool? isSubscription, bool? isExternal)
        {
            if (isSubscription == null)
                _purchaseTracker.TrackTrustedPurchase(record);
            else if (isSubscription is true)
                TrackExternalTrustedSubscriptionActivation(record);
            else TrackExternalTrustedInAppFor(record);
        }

        private void TrackExternalTrustedInAppFor([NotNull] TrustedPurchaseRecord record)
        {
            _analyticsService.TrackInApp(record.ProductId, record.Price, record.Currency, record.TransactionId, record.OriginalTransactionId, record.StoreName, new Option<string>(record.StoreFront));
            _campaignsTracker.TrackInAppPurchase(record.ProductId, InAppSourceKind.External);
            _purchaseTracker.TrackTrustedPurchase(record);
        }

        private void TrackExternalTrustedSubscriptionActivation([NotNull] TrustedPurchaseRecord record)
        {
            _analyticsService.TrackSubscription(record.PeriodType is PeriodType.Trial, record.ProductId, record.Price, record.Currency, record.TransactionId, record.OriginalTransactionId, record.StoreName, new Option<string>(record.StoreFront));
            _campaignsTracker.TrackSubscriptionPurchase(record.ProductId, InAppSourceKind.External);
            _purchaseTracker.TrackTrustedPurchase(record);
        }

        public bool IsApplicationInstalled([CanBeNull] string identifier)
        {
            return _platform.IsApplicationInstalled(identifier);
        }

        public void Reset(bool clearNativeStorage = true, bool clearCloudStorage = true)
        {
            _logger.Log("MagifyClient reset.");
            _generalPrefs.Reset();
            _appStatePrefs.Reset();
            if (clearNativeStorage) ClearNativeStorage();
            if (clearCloudStorage) _platform.RemoveValueFromCloudStorage(CloudStorageKeys.ClientId);
        }

        public void ClearNativeStorage()
        {
            _logger.Log("MagifyClient clear native storage.");
            _platform.ClearNativeStorage();
        }

        public void Dispose()
        {
            try
            {
                _disposables.Release();
                typeof(MinimalMagifyClient).ResetEvents(this);
            }
            catch (Exception e)
            {
                _logger.LogError($"During dispose of MagifyClient an error occurred: {e}");
            }
        }

        private void PrepareForFirstSessionInVersion()
        {
            _logger.Log($"Start of preparing for the current App version: {_appVersionProvider.AppVersion}.");
            using (_generalPrefs.MultipleChangeScope())
            {
                _logger.Log("Authorization Token will be reset.");
                _serverApi.ResetAuthorizationToken();
                _logger.Log("Notifying analytics session tracker.");
                _analyticsSessionTracker.TrackFirstSessionInVersion();
            }
            _logger.Log("Preparing for the current App version finished.");
        }

        private void AssertProductionTweak([NotNull] string tweakName)
        {
            if (Environment.Value is Magify.Environment.Production)
            {
                _logger.LogWarning($"Attempting to tweak the {tweakName} in production mode. This is not recommended as it may break user segmentation.");
            }
        }
    }
}