﻿using JetBrains.Annotations;
using Magify.Types;
using UnityEngine;
using EventType = Magify.Model.EventType;

namespace Magify
{
    internal class MinimalAnalyticsService
    {
        private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Analytics);

        private readonly MinimalAnalyticsTracker _analyticsTracker;

        public MinimalAnalyticsService(MinimalAnalyticsTracker analyticsTracker)
        {
            _analyticsTracker = analyticsTracker;
        }

        public void TrackAdsImpression(IAdsImpression adsImpression)
        {
            switch (adsImpression)
            {
                case ApplovinAdsImpression applovinData:
                    _analyticsTracker.TrackApplovinAdsImpression(applovinData);
                    break;
                case IronSourceAdsImpression ironSourceData:
                    _analyticsTracker.TrackIronSourceAdsImpression(ironSourceData);
                    break;
                default:
                    _logger.LogError($"Can't track ads impression event. Unknown impression data: {JsonFacade.SerializeObject(adsImpression)}");
                    break;
            }
        }

        public void TrackInApp(string productId, string price, string currency, string transactionId, string originalTransactionId, PurchaseStore store, [CanBeNull] Option<string> customStoreFront)
        {
            var product = new ProductInfo(productId, price, currency, store);
            var purchase = new PurchaseInfo
            {
                TransactionId = transactionId,
                OriginalTransactionId = originalTransactionId,
                Product = product,
                CustomStoreFront = customStoreFront,
            };
            _analyticsTracker.TrackInAppPurchaseEvent(EventType.InApp, purchase);
        }

        public void TrackSubscription(bool isTrial, string productId, string price, string currency, [CanBeNull] string transactionId, string originalTransactionId, PurchaseStore store, [CanBeNull] Option<string> customStoreFront)
        {
            var eventType = isTrial ? EventType.TrialActivation : EventType.PaidSubscriptionActivation;
            var product = new ProductInfo(productId, price, currency, store);
            var purchase = new PurchaseInfo
            {
                TransactionId = transactionId,
                OriginalTransactionId = originalTransactionId,
                Product = product,
                CustomStoreFront = customStoreFront,
            };
            _analyticsTracker.TrackSubscriptionPurchaseEvent(eventType, purchase);
        }

        public void SetupAnalyticsConfig(int eventsGroupSize, int syncTimeInterval)
        {
            var analyticsConfiguration = new AnalyticsConfiguration
            {
                GroupSize = eventsGroupSize,
                SyncIntervalInSeconds = syncTimeInterval,
            };
            _logger.Log($"Setup analytics config: ${analyticsConfiguration}");
            _analyticsTracker.SetupAnalyticsConfig(analyticsConfiguration);
        }

        public void ResetAnalyticsConfig()
        {
            _logger.Log("Reset analytics config");
            _analyticsTracker.ResetAnalyticsConfig();
        }
    }
}