﻿using System;
using System.Collections.Generic;
using System.Linq;
using JetBrains.Annotations;
using Magify.Model;
using Magify.Rx;
using UnityEngine;

namespace Magify
{
    internal class CampaignUpdateHandler : IInitializable, IForegroundListener, IDisposable
    {
        private static readonly MagifyLogger _logger = MagifyLogger.Get();

        private readonly Dictionary<string, Action> _campaignsChangingListeners = new();
        private readonly CampaignsProvider _campaignsProvider;
        private readonly CampaignsCollection _campaignsCollection;
        private readonly PlatformAPI _platform;
        private readonly CampaignsTracker _campaignsTracker;
        [NotNull]
        private readonly PooledCompositeDisposable _disposables = new();
        private readonly GeneralPrefs _prefs;
        private readonly LtoCampaignManager _ltoCampaignManager;

        public CampaignUpdateHandler(CampaignsTracker campaignsTracker, CampaignsProvider campaignsProvider, CampaignsCollection campaignsCollection, PlatformAPI platform, GeneralPrefs prefs, LtoCampaignManager ltoCampaignManager)
        {
            _campaignsTracker = campaignsTracker;
            _campaignsProvider = campaignsProvider;
            _campaignsCollection = campaignsCollection;
            _platform = platform;
            _prefs = prefs;
            _ltoCampaignManager = ltoCampaignManager;
        }

        void IInitializable.Initialize()
        {
            _platform.OnNewPackageInstalled
                .Subscribe(OnNewPackageInstalled)
                .AddTo(_disposables);
        }

        void IDisposable.Dispose()
        {
            _campaignsChangingListeners.Clear();
            _disposables.Release();
        }

        void IForegroundListener.OnForeground()
        {
            CheckActiveCampaignsUpdated();
        }

        public void SubscribeCampaignUpdates(string campaignName, Action onUpdate)
        {
            if (_campaignsChangingListeners.TryGetValue(campaignName, out var listeners))
            {
                onUpdate += listeners;
            }
            _campaignsChangingListeners[campaignName] = onUpdate;
        }

        public void UnsubscribeCampaignUpdates(string campaignName, Action onUpdate)
        {
            if (_campaignsChangingListeners.TryGetValue(campaignName, out var listeners))
            {
                _campaignsChangingListeners[campaignName] = listeners - onUpdate;
            }
        }

        public void ClearSubscriptions(string campaignName)
        {
            _campaignsChangingListeners.Remove(campaignName);
        }

        private void CheckActiveCampaignsUpdated()
        {
            _logger.Log($"CampaignUpdateHandler: {nameof(CheckActiveCampaignsUpdated)}, activeCampaigns: {_campaignsChangingListeners}");
            var nestedCampaigns = SelectActiveNestedCampaigns(p => !_campaignsProvider.IsNestedCampaignRelevant(p));
            foreach (var nestedCampaign in nestedCampaigns)
            {
                OnProductUsed(nestedCampaign.ProductId);
            }
        }

        private void OnNewPackageInstalled(string packageName)
        {
            _logger.Log($"CampaignUpdateHandler: {nameof(OnNewPackageInstalled)}, packageName: {packageName}");
            var nestedCampaigns = SelectActiveNestedCampaigns(p => p.PromotedApplication?.BundleId == packageName);
            foreach (var nestedCampaign in nestedCampaigns)
            {
                OnProductUsed(nestedCampaign.ProductId);
            }
        }

        private IEnumerable<NestedCampaign> SelectActiveNestedCampaigns(Func<NestedCampaign, bool> predicate)
        {
            var campaigns = _campaignsChangingListeners
                .Keys
                .Select(campaignName => _campaignsTracker.GetCampaignRequest(campaignName))
                .Where(campaignRequest => campaignRequest != null && campaignRequest.Campaign.IsSupportNested());
            return campaigns
                .Select(campaignRequest => campaignRequest.NestedCampaigns?.FirstOrDefault(predicate))
                .Where(nested => nested != null);
        }

        public void OnProductUsed(string productId)
        {
            _logger.Log($"CampaignUpdateHandler: {nameof(OnProductUsed)} productId: {productId}");
            var request = _campaignsTracker.GetCampaignRequestForProduct(productId);
            if (request == null)
            {
                return;
            }

            var campaign = request.Campaign;
            var actualCampaigns = request.IsDefaultCampaign ? _campaignsCollection.DefaultCampaigns : _campaignsCollection.CurrentCampaigns;
            var actualRecord = actualCampaigns.FirstOrDefault(c => c.Name == campaign.Name);
            if (actualRecord == null)
            {
                _logger.LogError($"Cannot find valid campaign for \"{campaign.Name}\" campaign name.");
                return;
            }
            var actualNestedCampaigns = _campaignsProvider.GetActiveNestedCampaigns(actualRecord.Campaign);
            if (request.NestedCampaigns.EqualsInAnyOrder(actualNestedCampaigns))
            {
                return;
            }
            if (actualNestedCampaigns?.Count == 0 && actualRecord.Campaign.Type?.IsLimitedOffer() is true)
            {
                _ltoCampaignManager.CompleteOffer(actualRecord.Campaign.Name);
            }

            var updatedProducts = actualNestedCampaigns?
                .Select(c => CampaignParser.ProductsParsers[c.Type!.Value].ParseProduct(c, _prefs.ClientId.Value))
                .Where(c => c != null);

            request.ParsedProducts!.PullStateFrom(updatedProducts!.ToList());
            if (_campaignsChangingListeners.TryGetValue(campaign.Name, out var listener))
            {
                listener?.Invoke();
            }
        }
    }
}