using System;
using System.Collections.Generic;

namespace Magify
{
    internal static class CampaignInfoExtensions
    {
        internal static bool IsWithinClickLimits(this CampaignInfo campaign, Counter<ClicksCounterScope> counter)
        {
            return campaign.ClickLimit == null || counter[ClicksCounterScope.Global, CounterKey.GetKey(campaignName: campaign.Name)] < campaign.ClickLimit;
        }

        internal static bool IsWithinRewardsGrantLimits(this CampaignInfo campaign, Counter<RewardsCounterScope> counter)
        {
            var global = campaign.RewardGrantLimits.Global == null || campaign.RewardGrantLimits.Global > counter[RewardsCounterScope.Global, CounterKey.GetKey(campaignName: campaign.Name)];
            var session = campaign.RewardGrantLimits.Session == null || campaign.RewardGrantLimits.Session > counter[RewardsCounterScope.Session, CounterKey.GetKey(campaignName: campaign.Name)];
            var daily = campaign.RewardGrantLimits.Daily == null || campaign.RewardGrantLimits.Daily > counter[RewardsCounterScope.Daily, CounterKey.GetKey(campaignName: campaign.Name)];

            return global && session && daily;
        }

        internal static bool IsWithinBonusGrantLimits(this CampaignInfo campaign, Counter<BonusesCounterScope> counter)
        {
            var global = campaign.BonusGrantLimits.Global == null || campaign.BonusGrantLimits.Global > counter[BonusesCounterScope.Global, CounterKey.GetKey(campaignName: campaign.Name)];
            var session = campaign.BonusGrantLimits.Session == null || campaign.BonusGrantLimits.Session > counter[BonusesCounterScope.Session, CounterKey.GetKey(campaignName: campaign.Name)];
            var daily = campaign.BonusGrantLimits.Daily == null || campaign.BonusGrantLimits.Daily > counter[BonusesCounterScope.Daily, CounterKey.GetKey(campaignName: campaign.Name)];

            return global && session && daily;
        }

        internal static bool IsWithinCampaignImpressionLimits(this CampaignInfo campaign, Counter<ImpressionsCounterScope> counter, IReadOnlyDictionary<string, List<DateTime>> impressionsTime)
        {
            var global = IsWithinImpressionLimit(counter, ImpressionsCounterScope.Global, campaign.Name, campaign.ImpressionLimits?.GlobalLimit);
            var session = IsWithinImpressionLimit(counter, ImpressionsCounterScope.Session, campaign.Name, campaign.ImpressionLimits?.SessionLimit);
            var daily = IsWithinImpressionLimit(counter, ImpressionsCounterScope.Daily, campaign.Name, campaign.ImpressionLimits?.DailyLimit);
            var period = campaign.IsWithinPeriodImpressionsLimit(impressionsTime);

            return global && session && daily && period;
        }

        internal static bool IsWithinPeriodImpressionsLimit(this CampaignInfo campaign, IReadOnlyDictionary<string, List<DateTime>> impressionsTime)
        {
            if (campaign.ImpressionLimits?.PeriodLimit == null || campaign.ImpressionLimits.Period == null)
            {
                return true;
            }

            var period = campaign.ImpressionLimits.Period.Value;
            var triggerImpressions = ImpressionUtils.GetImpressionsPerPeriod(impressionsTime, CounterKey.GetKeyCampaignBySourceType(campaign.Name, SourceType.Trigger).ToString(), period);
            var eventImpressions = ImpressionUtils.GetImpressionsPerPeriod(impressionsTime, CounterKey.GetKeyCampaignBySourceType(campaign.Name, SourceType.Event).ToString(), period);
            return eventImpressions + triggerImpressions < campaign.ImpressionLimits.PeriodLimit;
        }

        private static bool IsWithinImpressionLimit(Counter<ImpressionsCounterScope> counter, ImpressionsCounterScope scope, string campaignName, int? limitValue)
        {
            if (limitValue == null)
            {
                return true;
            }

            var triggerImpressions = counter[scope, CounterKey.GetKeyCampaignBySourceType(campaignName, SourceType.Trigger)];
            var eventImpressions = counter[scope, CounterKey.GetKeyCampaignBySourceType(campaignName, SourceType.Event)];
            return eventImpressions + triggerImpressions < limitValue;
        }
    }
}