using System;
using System.Collections.Generic;
using System.Linq;
using JetBrains.Annotations;
using Magify.Model;

namespace Magify
{
    internal class DefaultProductsContainer : IContextListener
    {
        [CanBeNull]
        private List<InAppProduct> _inAppProducts;
        [CanBeNull]
        private List<SubscriptionProduct> _subscriptionProducts;
        [CanBeNull]
        private List<DefaultProduct> _defaultProducts;
        [CanBeNull]
        internal List<DefaultProduct> DefaultProducts => _defaultProducts;
        [NotNull]
        private readonly object _lock = new();

        public ConfigScope SuitableScope => ConfigScope.ProductIds;

        [NotNull]
        public IReadOnlyList<InAppProduct> InAppProducts
        {
            get
            {
                lock (_lock)
                {
                    if (_defaultProducts == null)
                    {
                        return Array.Empty<InAppProduct>();
                    }
                    _inAppProducts ??= _defaultProducts
                        .Where(c => c.Type is ProductType.Consumable or ProductType.NonConsumable)
                        .SelectMany(mapByStore)
                        .ToList();
                    return _inAppProducts;
                }

                IEnumerable<InAppProduct> mapByStore([NotNull] DefaultProduct product)
                {
                    if (product.Stores == null)
                    {
                        yield return new InAppProduct(product.ProductId, product.Type == ProductType.Consumable, PurchaseStore.Native, null, product.ProductContext);
                        yield break;
                    }
                    foreach (var store in product.Stores)
                    {
                        yield return new InAppProduct(product.ProductId, product.Type == ProductType.Consumable, store, null, product.ProductContext);
                    }
                }
            }
        }

        [NotNull]
        public IReadOnlyList<SubscriptionProduct> SubscriptionProducts
        {
            get
            {
                lock (_lock)
                {
                    if (_defaultProducts == null)
                    {
                        return Array.Empty<SubscriptionProduct>();
                    }
                    _subscriptionProducts ??= _defaultProducts
                        .Where(c => c.Type is ProductType.Subscription)
                        .Select(c => new SubscriptionProduct(c.ProductId, null, c.ProductContext))
                        .ToList();
                    return _subscriptionProducts;
                }
            }
        }

        public void UpdateContext(CampaignsContext context, ContextKind kind)
        {
            lock (_lock)
            {
                switch (kind)
                {
                    case ContextKind.Default when _defaultProducts?.Count is null or 0 && _subscriptionProducts?.Count is null or 0:
                    case ContextKind.Saved or ContextKind.Downloaded:
                        _inAppProducts = null;
                        _subscriptionProducts = null;
                        _defaultProducts = context.DefaultProducts;
                        break;
                }
            }
        }

        public void ResetDefaultProducts()
        {
            lock (_lock)
            {
                _inAppProducts = null;
                _subscriptionProducts = null;
                _defaultProducts = null;
            }
        }
    }
}