using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using Cysharp.Threading.Tasks;
using Magify.Model;
using JetBrains.Annotations;
using Magify.Rx;
using UnityEngine;

namespace Magify
{
    internal class CampaignsTracker : MinimalCampaignsTracker, IInitializable, IForegroundListener, IDisposable
    {
        private static readonly MagifyLogger _logger = MagifyLogger.Get();

        private readonly GeneralPrefs _generalPrefs;
        private readonly Counters _counters;
        private readonly LimitsHolder _limitsHolder;
        private readonly PlatformAPI _platform;

        private readonly Dictionary<string, CampaignRequest> _campaignRequests = new();
        private readonly Dictionary<CampaignType, CampaignRequest> _campaignRequestsByType = new();
        private readonly List<(CampaignRequest Request, DateTime Time)> _campaignRequestTime = new();
        private readonly List<(CampaignType Type, DateTime Time)> _adsImpressionTime = new();

        private readonly Dictionary<string, List<DateTime>> _impressionsTime = new();
        private readonly Dictionary<string, string> _clickedCampaignByProduct = new();
        private readonly Dictionary<string, string> _clickedNestedByCampaign = new();
        [NotNull]
        private readonly PooledCompositeDisposable _disposables = new();
        private CancellationTokenSource _dailyResetCancellationToken;

        public IReadOnlyDictionary<string, List<DateTime>> ImpressionsTime => _impressionsTime;

        public ICollection<string> AllUsedProducts => _generalPrefs.AllUsedProducts;

        public override event Action OnPurchasedProductsChanged
        {
            add => base.OnPurchasedProductsChanged += value;
            remove => base.OnPurchasedProductsChanged -= value;
        }

        public CampaignsTracker(GeneralPrefs generalPrefs, Counters counters, LimitsHolder limitsHolder, PlatformAPI platformAPI)
            : base(generalPrefs)
        {
            _generalPrefs = generalPrefs;
            _counters = counters;
            _limitsHolder = limitsHolder;
            _platform = platformAPI;
        }

        void IInitializable.Initialize()
        {
            _generalPrefs.GlobalSessionCounter
                .Subscribe(_ => ResetSessionCounters())
                .AddTo(_disposables);

            ResetDailyCountersIfNeeded();
            ScheduleDailyCountersReset();
        }

        void IForegroundListener.OnForeground()
        {
            ResetDailyCountersIfNeeded();
        }

        void IDisposable.Dispose()
        {
            _disposables.Release();

            _dailyResetCancellationToken?.Cancel();
            _dailyResetCancellationToken?.Dispose();
        }

        [CanBeNull]
        public CampaignRequest GetCampaignRequest([CanBeNull] string campaignName)
        {
            return campaignName == null ? null : _campaignRequests.GetValueOrDefault(campaignName);
        }

        [CanBeNull]
        public CampaignRequest GetCampaignRequest(CampaignType campaignType)
        {
            return _campaignRequestsByType.GetValueOrDefault(campaignType);
        }

        [CanBeNull]
        public string GetNestedForCampaign(string campaignName)
        {
            return campaignName == null ? null : _clickedNestedByCampaign.GetValueOrDefault(campaignName, null);
        }

        [CanBeNull]
        public CampaignRequest GetCampaignRequestForProduct(string productId)
        {
            var campaignName = productId == null ? null : _clickedCampaignByProduct.GetValueOrDefault(productId, null);
            return campaignName != null ? GetCampaignRequest(campaignName) : null;
        }

        [CanBeNull]
        public CampaignImpression CreateCampaignImpression([CanBeNull] string campaignName, [CanBeNull] string productId = null, [CanBeNull] string productIdCreative = null)
        {
            var request = GetCampaignRequest(campaignName);
            if (request == null)
            {
                return null;
            }
            return CreateCampaignImpression(campaignName, request, productId, productIdCreative);
        }

        [CanBeNull]
        public CampaignImpression GetNestedCampaignImpression([NotNull] string productId, [CanBeNull] PurchaseStore? store = null)
        {
            var request = GetCampaignRequestForProduct(productId);
            if (request == null) return null;
            if (!request.TryGetNestedByProduct(productId, store, out var nestedCampaign)) return null;
            var impression = CreateCampaignImpression(request.Campaign.Name, nestedCampaign.ProductId, nestedCampaign.ProductIdCreative);
            if (impression == null) return null;
            impression.CampaignName = nestedCampaign.Name;
            impression.StoreName = nestedCampaign.StoreName;
            return impression;
        }

        [CanBeNull]
        public CampaignImpression GetNestedAdsCampaignImpression(string campaignName)
        {
            var request = GetCampaignRequest(campaignName);
            if (request == null) return null;

            var nestedCampaignName = GetNestedForCampaign(campaignName);
            var nestedCampaignType = request.Campaign.Type.GetNestedCampaignType();

            if (nestedCampaignName == null || nestedCampaignType == null) return null;

            if (!request.TryGetNestedByName(nestedCampaignName, out var nestedCampaign)) return null;

            return CreateNestedCampaignImpression(
                nestedCampaign,
                nestedCampaignType,
                request.IsDefaultCampaign,
                request.Source.Name,
                request.CustomParams);
        }

        [NotNull]
        public IReadOnlyList<CampaignImpression> GetNestedCampaignImpressions([CanBeNull] string campaignName, [NotNull] IEnumerable<string> productIds)
        {
            var request = GetCampaignRequest(campaignName);
            if (request == null) return Array.Empty<CampaignImpression>();

            if (request.Campaign.IsSupportNested())
            {
                return request.NestedCampaigns!
                    .Where(p => productIds.Contains(p.ProductId))
                    .Select(p => CreateNestedCampaignImpression(campaignName, request, p))
                    .ToList();
            }

            return Array.Empty<CampaignImpression>();
        }

        public CampaignImpression BuildExternalInAppImpression(string productId)
        {
            return CampaignImpression.BuildExternalInAppImpression(productId, _generalPrefs.GlobalSessionCounter.Value);
        }

        public CampaignImpression BuildExternalSubscriptionImpression(string productId)
        {
            return CampaignImpression.BuildExternalSubscriptionImpression(productId, _generalPrefs.GlobalSessionCounter.Value);
        }

        public DateTime? GetLastImpressionTimestampByLimitedSource(CampaignType? type)
        {
            for (var i = _campaignRequestTime.Count - 1; i >= 0; i--)
            {
                var pair = _campaignRequestTime[i];
                if (!pair.Request.Source.Type.HasImpressionLimits())
                    continue;

                if (type == null || pair.Request.Campaign.Type == type)
                    return pair.Time;
            }
            return null;
        }

        public DateTime? GetLastAdsImpressionTimestamp(CampaignType[] types)
        {
            for (var i = _adsImpressionTime.Count - 1; i >= 0; i--)
            {
                var pair = _adsImpressionTime[i];
                if (types.Contains(pair.Type))
                {
                    return pair.Time;
                }
            }
            return null;
        }

        public DateTime? GetRelativeImpressionTimestamp(CampaignType type)
        {
            return type switch
            {
                CampaignType.Interstitial => GetLastAdsImpressionTimestamp(CampaignTypeUtils.RewardedCampaignTypes),
                _ => null
            };
        }

        public void RegisterCampaignRequest(CampaignRequest request)
        {
            _campaignRequests[request.Campaign.Name] = request;
            _campaignRequestsByType[request.Campaign.Type] = request;
        }

        public void TrackImpression(string campaignName)
        {
            var request = GetCampaignRequest(campaignName);
            if (request == null)
            {
                _logger.LogError($"Can't process {nameof(TrackImpression)} for campaign={campaignName}, since requested campaign was not found.");
                return;
            }

            var keyCampaignBySource = CounterKey.GetKeyCampaignBySource(campaignName, request.Source.Name, request.Source.Type);
            var keyCampaignBySourceType = CounterKey.GetKeyCampaignBySourceType(campaignName, request.Source.Type);
            var keyCampaignTypeBySourceType = CounterKey.GetKeyCampaignTypeBySourceType(request.Campaign.Type, request.Source.Type);

            using (_counters.Impressions.MultipleChangeScope())
            {
                ImpressionsCounterScope? excludeScope = request.Campaign.Type.IsLimitedOffer() is false ? ImpressionsCounterScope.Activation : null;

                _counters.Impressions.IncrementAll(CounterKey.GetKey(campaignName: campaignName), excludeScope);
                _counters.Impressions.IncrementAll(keyCampaignBySource, excludeScope);
                _counters.Impressions.IncrementAll(keyCampaignBySourceType, excludeScope);

                _counters.Impressions.IncrementAll(keyCampaignTypeBySourceType, ImpressionsCounterScope.Activation);
                _counters.Impressions.IncrementAll(CounterKey.GetKey(campaignType: request.Campaign.Type), ImpressionsCounterScope.Activation);
                _counters.Impressions.IncrementAll(CounterKey.GetKey(sourceType: request.Source.Type), ImpressionsCounterScope.Activation);

                if (request.Source.Type.HasImpressionLimits() && _limitsHolder.GetActiveLimits().CampaignTypeLimits.TryGetValue(request.Campaign.Type, out var campaignLimits))
                {
                    foreach (var limit in campaignLimits.TempLimits)
                    {
                        _counters.Impressions.IncrementAll(CounterKey.GetKeyCampaignTypeByPeriodLimit(request.Campaign.Type, limit.LimitId), ImpressionsCounterScope.Activation);
                    }
                }
            }

            var impressionTime = DateTime.UtcNow;
            _campaignRequestTime.Add((request, impressionTime));

            SaveImpressionTimestamp(keyCampaignBySource.ToString(), impressionTime);
            SaveImpressionTimestamp(keyCampaignBySourceType.ToString(), impressionTime);
            SaveImpressionTimestamp(keyCampaignTypeBySourceType.ToString(), impressionTime);
        }

        public void TrackAdsImpression(string campaignName)
        {
            var request = GetCampaignRequest(campaignName);
            if (request == null)
            {
                _logger.LogError($"Can't process {nameof(TrackAdsImpression)} for campaign={campaignName}, since requested campaign was not found.");
                return;
            }

            if (request.Campaign.Type.HasNestedAdsCampaigns())
            {
                TrackNestedAdsImpression(request);
            }
            else
            {
                TrackImpression(campaignName);
            }
            _adsImpressionTime.Add((request.Campaign.Type, DateTime.UtcNow));
        }

        private void TrackNestedAdsImpression(CampaignRequest campaignRequest)
        {
            var nestedCampaignName = GetNestedForCampaign(campaignRequest.Campaign.Name);
            var nestedCampaignType = campaignRequest.Campaign.Type.GetNestedCampaignType();

            if (nestedCampaignName == null || nestedCampaignType == null) return;

            using (_counters.Impressions.MultipleChangeScope())
            {
                ImpressionsCounterScope? excludeScope = campaignRequest.Campaign.Type.IsLimitedOffer() is false ? ImpressionsCounterScope.Activation : null;
                _counters.Impressions.IncrementAll(CounterKey.GetKey(nestedName: nestedCampaignName), excludeScope);
                _counters.Impressions.IncrementAll(CounterKey.GetKey(nestedType: nestedCampaignType), ImpressionsCounterScope.Activation);
            }
        }

        public void TrackEvent(string eventName)
        {
            _counters.Events.IncrementAll(CounterKey.GetKey(source: eventName));
        }

        public void RevertEvent(string eventName)
        {
            _counters.Events.DecrementAll(CounterKey.GetKey(source: eventName));
        }

        public void TrackClick(string campaignName, [CanBeNull] string productId = null, [CanBeNull] PurchaseStore? store = null)
        {
            var request = GetCampaignRequest(campaignName);
            if (request == null)
            {
                _logger.LogError($"Can't process {nameof(TrackClick)} for campaign={campaignName}, since requested campaign was not found.");
                return;
            }

            TrackCampaignClick(request, productId);
            if (productId != null)
            {
                _clickedCampaignByProduct[productId] = campaignName;
                if (request.TryGetNestedByProduct(productId, store, out var nestedCampaign))
                {
                    _clickedNestedByCampaign[campaignName] = nestedCampaign.Name;
                }
            }

            using (_counters.Clicks.MultipleChangeScope())
            {
                _counters.Clicks.IncrementAll(CounterKey.GetKey(campaignName: request.Campaign.Name));
                _counters.Clicks.IncrementAll(CounterKey.GetKeyCampaignBySource(campaignName, request.Source.Name, request.Source.Type));
            }
        }

        private void TrackCampaignClick(CampaignRequest campaignRequest, string productId)
        {
            // productId will be null only for fake cross promo products
            // delete this when real cross promo products will be added to CrossPromoCampaign
            if (campaignRequest.Campaign is CrossPromoCampaign promoCampaign && productId is null)
            {
                _platform.TrackPromotedApplication(promoCampaign.Products.FirstOrDefault(c => c.Id == ProductDef.FakeProductId)?.BundleId);
            }

            if (productId != null && campaignRequest.TryGetNestedByProduct(productId, store: null, out var nestedCampaign) && nestedCampaign.Type == ProductType.CrossPromo)
            {
                _platform.TrackPromotedApplication(nestedCampaign.PromotedApplication?.BundleId);
            }
        }

        public override void TrackInAppPurchase(string productId, InAppSourceKind sourceKind)
        {
            base.TrackInAppPurchase(productId, sourceKind);
            if (sourceKind is InAppSourceKind.Internal)
            {
                TrackNestedCampaignUsed(productId);
            }
        }

        public override void TrackSubscriptionPurchase(string productId, InAppSourceKind sourceKind)
        {
            base.TrackSubscriptionPurchase(productId, sourceKind);
            if (sourceKind is InAppSourceKind.Internal)
            {
                TrackNestedCampaignUsed(productId);
            }
        }

        public void TrackRewardGranted(string productId)
        {
            UpdateProductReceiptInfo(productId);
            TrackNestedCampaignUsed(productId);
            _generalPrefs.UsedFreeProducts.Add(productId);

            var request = GetCampaignRequestForProduct(productId);
            if (request == null)
            {
                _logger.LogError($"Can't track reward. Campaign for product was not found: {productId}");
                return;
            }
            using (_counters.Rewards.MultipleChangeScope())
            {
                RewardsCounterScope? excludedScope = request.Campaign.Type.IsLimitedOffer() is false ? RewardsCounterScope.Activation : null;
                _counters.Rewards.IncrementAll(CounterKey.GetKey(campaignName: request.Campaign.Name), excludedScope);
                var keyCampaignBySource = CounterKey.GetKeyCampaignBySource(request.Campaign.Name, request.Source.Name, request.Source.Type);
                _counters.Rewards.IncrementAll(keyCampaignBySource, excludedScope);
            }
        }

        public void TrackBonusGranted(string productId)
        {
            UpdateProductReceiptInfo(productId);
            TrackNestedCampaignUsed(productId);
            _generalPrefs.UsedFreeProducts.Add(productId);

            var request = GetCampaignRequestForProduct(productId);
            if (request == null)
            {
                _logger.LogError($"Can't track bonus. Campaign for product was not found: {productId}");
                return;
            }
            using (_counters.Bonuses.MultipleChangeScope())
            {
                _counters.Bonuses.IncrementAll(CounterKey.GetKey(campaignName: request.Campaign.Name));
                var keyCampaignBySource = CounterKey.GetKeyCampaignBySource(request.Campaign.Name, request.Source.Name, request.Source.Type);
                _counters.Bonuses.IncrementAll(keyCampaignBySource);
            }
        }

        public void TrackOrdinaryProductUsed(string productId)
        {
            UpdateProductReceiptInfo(productId);
            TrackNestedCampaignUsed(productId);
            _generalPrefs.UsedFreeProducts.Add(productId);
        }

        private void SaveImpressionTimestamp(string key, DateTime time)
        {
            var values = _impressionsTime.GetValueOrDefault(key, new List<DateTime>());
            values.Add(time);
            _impressionsTime[key] = values;
        }

        protected override void UpdateProductReceiptInfo(string productId)
        {
            base.UpdateProductReceiptInfo(productId);

            var (temporaryLimits, @lock) = _limitsHolder.IterateTemporaryLimitsForProductWithLock(productId);
            if (temporaryLimits == null || @lock == null)
            {
                _logger.LogError($"Something went wrong while updating product receipt: {productId}");
                return;
            }
            using (_counters.Impressions.MultipleChangeScope())
            {
                lock (@lock)
                {
                    foreach (var limit in temporaryLimits)
                    {
                        foreach (var type in EnumExtensions.GetValues<CampaignType>())
                        {
                            _counters.Impressions.Reset(ImpressionsCounterScope.Period, CounterKey.GetKeyCampaignTypeByPeriodLimit(type, limit.GroupId));
                        }
                    }
                }
            }
        }

        private void TrackNestedCampaignUsed([NotNull] string productId, [CanBeNull] PurchaseStore? store = null)
        {
            var request = GetCampaignRequestForProduct(productId);
            if (request == null)
            {
                _logger.LogError($"Can't find campaign for product: {productId}");
                return;
            }

            if (request.TryGetNestedByProduct(productId, store, out var nestedCampaign))
            {
                NestedCounterScope? excludeScope = request.Campaign.Type.IsLimitedOffer() is false ? NestedCounterScope.Activation : null;
                _counters.Nested.IncrementAll(CounterKey.GetKey(nestedName: nestedCampaign.Name), excludeScope);
            }
        }

        private CampaignImpression CreateCampaignImpression(string campaignName, CampaignRequest request, [CanBeNull] string productId, [CanBeNull] string productIdCreative)
        {
            var campaignType = request.Campaign.Type;
            var isDefaultCampaign = request.IsDefaultCampaign;
            var eventName = request.Source.Name;
            var eventParams = request.CustomParams;

            return new CampaignImpression
            {
                CampaignName = campaignName,
                IsDefaultConfig = isDefaultCampaign,
                ImpressionNumber = _counters.Impressions[ImpressionsCounterScope.Global, CounterKey.GetKey(campaignName: campaignName)],
                SessionImpressionNumber = _counters.Impressions[ImpressionsCounterScope.Session, CounterKey.GetKey(campaignName: campaignName)],
                CampaignTypeImpressionNumber = _counters.Impressions[ImpressionsCounterScope.Global, CounterKey.GetKey(campaignType: campaignType)],
                CampaignTypeSessionImpressionNumber = _counters.Impressions[ImpressionsCounterScope.Session, CounterKey.GetKey(campaignType: campaignType)],
                EventName = eventName,
                EventNumber = _counters.Events[EventsCounterScope.Global, CounterKey.GetKey(source: eventName)],
                SessionEventNumber = _counters.Events[EventsCounterScope.Session, CounterKey.GetKey(source: eventName)],
                SessionNumber = _generalPrefs.GlobalSessionCounter.Value,
                ProductId = productId,
                ProductIdCreative = productIdCreative,
                StoreName = (request.Campaign as ICampaignWithPurchaseStore)?.Store,
                Parameters = eventParams,
            };
        }

        private CampaignImpression CreateNestedCampaignImpression(
            string parentCampaignName,
            CampaignRequest request,
            NestedCampaign nestedCampaign)
        {
            var impression = CreateCampaignImpression(parentCampaignName, request, nestedCampaign.ProductId, nestedCampaign.ProductIdCreative);
            impression.CampaignName = nestedCampaign.Name;
            impression.StoreName = nestedCampaign.StoreName;
            return impression;
        }

        private CampaignImpression CreateNestedCampaignImpression(
            NestedCampaign nestedCampaign,
            string nestedType,
            bool isDefaultCampaign,
            string eventName,
            Dictionary<string, object> eventParams)
        {
            return new CampaignImpression
            {
                CampaignName = nestedCampaign.Name,
                IsDefaultConfig = isDefaultCampaign,
                ImpressionNumber = _counters.Impressions[ImpressionsCounterScope.Global, CounterKey.GetKey(nestedName: nestedCampaign.Name)],
                SessionImpressionNumber = _counters.Impressions[ImpressionsCounterScope.Session, CounterKey.GetKey(nestedName: nestedCampaign.Name)],
                CampaignTypeImpressionNumber = _counters.Impressions[ImpressionsCounterScope.Global, CounterKey.GetKey(nestedType: nestedType)],
                CampaignTypeSessionImpressionNumber = _counters.Impressions[ImpressionsCounterScope.Session, CounterKey.GetKey(nestedType: nestedType)],
                EventName = eventName,
                EventNumber = _counters.Events[EventsCounterScope.Global, CounterKey.GetKey(source: eventName)],
                SessionEventNumber = _counters.Events[EventsCounterScope.Session, CounterKey.GetKey(source: eventName)],
                SessionNumber = _generalPrefs.GlobalSessionCounter.Value,
                ProductId = nestedCampaign.ProductId,
                ProductIdCreative = nestedCampaign.ProductIdCreative,
                StoreName = nestedCampaign.StoreName,
                Parameters = eventParams
            };
        }

        public void UpdateRelatedCounters()
        {
            ResetSessionCounters();
            ResetDailyCountersIfNeeded();
            ScheduleDailyCountersReset();

            if (_generalPrefs.VersionSessionCounter == 0)
            {
                ResetVersionCounters();
            }
        }

        private void ResetDailyCountersIfNeeded()
        {
            var nextDayStartTime = DateTime.Today.AddDays(1);
            if (_generalPrefs.NextDailyResetTime == default)
            {
                _generalPrefs.NextDailyResetTime = nextDayStartTime;
                return;
            }
            if (DateTime.Now < _generalPrefs.NextDailyResetTime) return;

            ResetDailyCounters();
            _generalPrefs.NextDailyResetTime = nextDayStartTime;
        }

        private void ScheduleDailyCountersReset()
        {
            _dailyResetCancellationToken?.Cancel();
            _dailyResetCancellationToken?.Dispose();
            _dailyResetCancellationToken = new CancellationTokenSource();

            try
            {
                var nextDayStartTime = DateTime.Today.AddDays(1);
                var delay = nextDayStartTime - DateTime.Now;
                UniTask
                    .Delay(delay, delayType: DelayType.Realtime, cancellationToken: _dailyResetCancellationToken.Token)
                    .ContinueWith(ResetDailyCountersIfNeeded);
            }
            catch (Exception e)
            {
                _logger.LogError($"Failed to schedule daily counters reset: {e}");
                _logger.LogException(e);
            }
        }

        internal void ResetGlobalCounters()
        {
            _counters.ResetAll(CounterScope.Global);
        }

        internal void ResetVersionCounters()
        {
            _counters.ResetAll(CounterScope.Version);
        }

        internal void ResetSessionCounters()
        {
            _counters.ResetAll(CounterScope.Session);
            _impressionsTime.Clear();
        }

        internal void ResetDailyCounters()
        {
            _counters.ResetAll(CounterScope.Daily);
        }

        internal void ResetActivationCounters(LtoModel model)
        {
            _counters.ResetAll(CounterScope.Activation, key => key.CampaignName == model.CampaignName || model.Campaign.NestedCampaigns?.Any(nested => nested.Name == key.NestedName) is true);
        }

        internal void ResetLtoStartCounter(LtoStartCounterScope scope, CounterKey key)
        {
            _counters.LtoStart.Reset(scope, key);
        }
    }
}