using System.Collections.Generic;
using System.Linq;
using Magify.Model;
using JetBrains.Annotations;
using UnityEngine;

namespace Magify
{
    internal static class CampaignMapper
    {
        private static readonly MagifyLogger _logger = MagifyLogger.Get();
        internal static readonly Dictionary<CampaignType, CampaignParser> Parsers = new()
        {
            { CampaignType.Banner, new BannerCampaignParser() },
            { CampaignType.Interstitial, new InterstitialCampaignParser() },
            { CampaignType.RewardedVideo, new RewardedVideoCampaignParser() },
            { CampaignType.LtoRewardedVideo, new LtoRewardedVideoCampaignParser() },
            { CampaignType.Bonus, new BonusCampaignParser() },
            { CampaignType.LtoBonus, new LtoBonusCampaignParser() },
            { CampaignType.InApp, new InAppCampaignParser() },
            { CampaignType.LtoInApp, new LtoInAppCampaignParser() },
            { CampaignType.Subscription, new SubscriptionCampaignParser() },
            { CampaignType.LtoSubscription, new LtoSubscriptionCampaignParser() },
            { CampaignType.Mixed, new MixedCampaignParser() },
            { CampaignType.LtoMixed, new LtoMixedCampaignParser() },
            { CampaignType.External, new ExternalPromoCampaignParser() },
            { CampaignType.LtoExternal, new LtoExternalPromoCampaignParser() },
            { CampaignType.CrossPromo, new CrossPromoCampaignParser() },
            { CampaignType.LtoCrossPromo, new LtoCrossPromoCampaignParser() },
            { CampaignType.Notification, new NotificationCampaignParser() },
            { CampaignType.RateReview, new RateReviewCampaignParser() },
        };

        [NotNull]
        public static List<CampaignRecord> MapCampaigns([NotNull] IEnumerable<CampaignModel> models)
        {
            var list = new List<CampaignRecord>();
            foreach (var model in models.Where(c => c != null))
            {
                var type = model.Type;
                if (type != null && Parsers.TryGetValue(type.Value, out var parser) && parser.IsValid(model))
                {
                    var campaignInfo = CampaignParser.ExtractCampaignInfo(model);
                    list.Add(new CampaignRecord(campaignInfo, model));
                }
                else
                {
                    _logger.LogWarning($"Unexpected campaign model. Name={model.Name}; Type={model.Type};");
                }
            }
            return list;
        }

        public static ParsedCampaign CreateCampaignFromModel([NotNull] this CampaignModel model, IEnumerable<NestedCampaign> filteredNestedCampaigns, CampaignParser.AdditionalData additionalData)
        {
            var type = model.Type;
            if (type == null || !Parsers.TryGetValue(type.Value, out var parser))
            {
                var campaignName = model.Name;
                var campaignType = model.Type?.ToString() ?? "null";
                throw new MagifyInvalidCampaignParserException(campaignName, campaignType);
            }

            return parser.Parse(model, filteredNestedCampaigns, additionalData);
        }
    }
}