using System;
using System.Collections.Generic;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Model;
using Magify.Rx;
using UnityEngine;

namespace Magify
{
    internal class ContextApplicator : IDisposable
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get();
        [NotNull]
        private readonly SubsystemsCollection _subsystems;
        [NotNull]
        private readonly object _lock = new();
        [NotNull]
        private readonly Subject<ContextKind> _contextApplied = new();
        [NotNull]
        private readonly Subject<ContextKind> _contextAppliedOnMainThread = new();
        [NotNull]
        private readonly Subject<ContextKind> _contextSkipped = new();
        [NotNull]
        private readonly Queue<ContextKind> _appliedContextsQueue = new();
        [NotNull]
        private readonly CancellationTokenSource _onDisposeCts = new();

        [CanBeNull]
        private CampaignsContext _waitingDefaultContext;
        private bool _isThereWaitingDefaultContext;
        [CanBeNull]
        private CampaignsContext _waitingDownloadedContext;
        private bool _isThereWaitingDownloadedContext;

        [NotNull]
        public IObservable<ContextKind> ContextApplied => _contextApplied;
        [NotNull]
        public IObservable<ContextKind> ContextAppliedOnMainThread => _contextAppliedOnMainThread;
        [NotNull]
        public IObservable<ContextKind> ContextSkipped => _contextSkipped;
        internal bool SavedContextApplied { get; private set; }
        internal bool DefaultContextApplied { get; private set; }
        internal bool DownloadedContextApplied { get; private set; }

        public ContextApplicator([NotNull] SubsystemsCollection subsystems)
        {
            _subsystems = subsystems;
        }

        public void Dispose()
        {
            _onDisposeCts.Cancel();
            _onDisposeCts.Dispose();

            lock (_lock)
            {
                SavedContextApplied = false;
                DefaultContextApplied = false;
                DownloadedContextApplied = false;
                _waitingDefaultContext = null;
                _isThereWaitingDefaultContext = false;
                _waitingDownloadedContext = null;
                _isThereWaitingDownloadedContext = false;
            }
        }

        public void HandleContext([CanBeNull] CampaignsContext context, ContextKind kind)
        {
            bool shouldApply;
            lock (_lock)
            {
                shouldApply = ShouldApply(kind);
            }
            if (shouldApply)
            {
                ApplyContext(context, kind);
                return;
            }

            lock (_lock)
            {
                switch (kind)
                {
                    case ContextKind.Default:
                        _waitingDefaultContext = context;
                        _isThereWaitingDefaultContext = true;
                        break;
                    case ContextKind.Downloaded:
                        _waitingDownloadedContext = context;
                        _isThereWaitingDownloadedContext = true;
                        break;
                    default:
                        throw new ArgumentOutOfRangeException($"Context of kind {kind} wasn't expected.");
                }
            }
        }

        private void ApplyContext([CanBeNull] CampaignsContext context, ContextKind kind)
        {
            var nextContext = default(CampaignsContext);
            var nextKind = default(ContextKind);
            lock (_lock)
            {
                if (context is not null)
                {
                    try
                    {
                        _subsystems.UpdateContextForAll(context, kind);
                        OnContextApplied(kind).Forget();
                    }
                    catch (Exception e)
                    {
                        _logger.LogError($"Exception during {kind} context applying: {e}");
                        _logger.LogException(e);
                    }
                }
                else
                {
                    _contextSkipped.OnNext(kind);
                }
                if (kind is ContextKind.Default && context == _waitingDefaultContext)
                {
                    _waitingDefaultContext = null;
                    _isThereWaitingDefaultContext = false;
                }
                if (kind is ContextKind.Downloaded && context == _waitingDownloadedContext)
                {
                    _waitingDownloadedContext = null;
                    _isThereWaitingDownloadedContext = false;
                }

                if (kind is ContextKind.Saved) SavedContextApplied = true;
                if (kind is ContextKind.Default) DefaultContextApplied = true;
                if (kind is ContextKind.Downloaded) DownloadedContextApplied = true;

                if (_isThereWaitingDefaultContext && ShouldApply(ContextKind.Default))
                {
                    nextContext = _waitingDefaultContext;
                    nextKind = ContextKind.Default;
                }
                else if (_isThereWaitingDownloadedContext && ShouldApply(ContextKind.Downloaded))
                {
                    nextContext = _waitingDownloadedContext;
                    nextKind = ContextKind.Downloaded;
                }
                else
                {
                    return;
                }
            }
            HandleContext(nextContext, nextKind);
        }

        private async UniTaskVoid OnContextApplied(ContextKind kind)
        {
            if (_onDisposeCts.IsCancellationRequested)
            {
                return;
            }

            _contextApplied.OnNext(kind);
            lock (_appliedContextsQueue)
            {
                _appliedContextsQueue.Enqueue(kind);
                if (_appliedContextsQueue.Count > 1)
                {
                    return;
                }
            }

            await TaskScheduler.SwitchToMainThread(_onDisposeCts.Token);
            lock (_appliedContextsQueue)
            {
                while (_appliedContextsQueue.TryDequeue(out kind))
                {
                    _contextAppliedOnMainThread.OnNext(kind);
                }
            }
        }

        private bool ShouldApply(ContextKind kind)
        {
            return kind switch
            {
                ContextKind.Saved => true,
                ContextKind.Default when SavedContextApplied => true,
                ContextKind.Downloaded when SavedContextApplied && DefaultContextApplied => true,
                _ => false,
            };
        }
    }
}