﻿using System;
using System.Text;
using JetBrains.Annotations;

namespace Magify
{
    internal class ToJsonToBase64MagifyEncoder : IMagifyEncoder
    {
        [NotNull]
        public string Encode<T>([NotNull] T value, [NotNull] string encodingModePrefix)
            where T: class
        {
            // To json
            var json = JsonFacade.SerializeObject(value);

            // To base64
            var stateBytes = Encoding.UTF8.GetBytes(json);
            // Calculate future base64 string size to use Convert.ToBase64CharArray() and avoid re-allocations
            var stateBytesLength = Base64Utils.CalculateAndValidateOutputLength(stateBytes.Length, false);
            var prefixLength = encodingModePrefix.Length;
            var result = new char[prefixLength + stateBytesLength];
            encodingModePrefix.CopyTo(0, result, 0, prefixLength); // Write prefix

            Convert.ToBase64CharArray(
                inArray: stateBytes, offsetIn: 0, stateBytes.Length, // Write all stateBytes...
                outArray: result, offsetOut: prefixLength); // with offset for prefix chars
            return new string(result);
        }

        [CanBeNull]
        public T Decode<T>([CanBeNull] string encoded, int encodingModePrefixLength)
            where T: class
        {
            if (string.IsNullOrEmpty(encoded))
            {
                return null;
            }

            // From base64
            var span = (ReadOnlySpan<char>)encoded;
            var bytes = new byte[span.Length - encodingModePrefixLength];
            Convert.TryFromBase64Chars(span[encodingModePrefixLength..], bytes, out _); // Ignore prefix and read app state bytes
            var json = Encoding.UTF8.GetString(bytes);

            // From json
            return JsonFacade.DeserializeObject<T>(json);
        }
    }
}