﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Model;
using UnityEngine;

namespace Magify
{
    internal class AppStateBuilder
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.AppState);

        [NotNull]
        private readonly GeneralPrefs _generalPrefs;
        [NotNull]
        private readonly CountersStorage _counters;
        [NotNull]
        private readonly ClientStateConfig _clientStateConfig;
        [NotNull]
        private readonly SdkByPlatformStateFileStorage _fileStorage;

        internal EncodingMode EncodingMode { get; set; } = EncodingMode.ToJsonToZipToBase64;

        public AppStateBuilder([NotNull] GeneralPrefs generalPrefs, [NotNull] CountersStorage counters, [NotNull] ClientStateConfig clientStateConfig, [NotNull] string storagePath)
        {
            _generalPrefs = generalPrefs;
            _counters = counters;
            _clientStateConfig = clientStateConfig;
            _fileStorage = new SdkByPlatformStateFileStorage(storagePath);
        }

        [NotNull, ItemNotNull]
        public async UniTask<string> Build(CancellationToken cancellationToken, RuntimePlatform platform)
        {
            var state = await CollectData(cancellationToken, platform);
            return DefaultMagifyEncoders.Get(EncodingMode).Encode(state, EncodingMode.ToPrefixString());
        }

        [NotNull, ItemNotNull]
        private async UniTask<ApplicationState> CollectData(CancellationToken cancellationToken, RuntimePlatform platform)
        {
            var clientState = default(string);
            var stateByPlatform = default(Dictionary<RuntimePlatform, ApplicationState.PlatformSdk>);
            try
            {
                if (_clientStateConfig.AppStateProvider != null)
                {
                    clientState = await _clientStateConfig.AppStateProvider.PrepareState(cancellationToken);
                }
                stateByPlatform = _fileStorage.LoadState();
                stateByPlatform[platform] = new ApplicationState.PlatformSdk(_generalPrefs.GetFileContent(), _counters.GetFileContent());
            }
            catch (Exception e)
            {
                var exception = new MagifyFailedToPrepareClientStateException(_clientStateConfig.AppStateProvider, e);
                _logger.LogException(exception);
                throw exception;
            }
            return new ApplicationState
            {
                ClientState = clientState,
                SdkByPlatform = stateByPlatform,
            };
        }

        [CanBeNull]
        public ApplicationState Parse([CanBeNull] string encoded, RuntimePlatform currentPlatform)
        {
            if (string.IsNullOrEmpty(encoded))
            {
                return null;
            }

            var encodingMode = EncodingMode.None.FromPrefixString(encoded);
            var state = DefaultMagifyEncoders.Get(encodingMode).Decode<ApplicationState>(encoded, EncodingModeUtils.EncodingModePrefixLength);
            if (state is { SdkByPlatform: { Count: > 0 } })
            {
                var others = state.SdkByPlatform.Where(s => s.Key != currentPlatform);
                var otherPlatforms = new Dictionary<RuntimePlatform, ApplicationState.PlatformSdk>(others);
                _fileStorage.SaveState(otherPlatforms);
            }
            return state;
        }
    }
}