using System;
using System.Collections.Generic;
using System.Linq;
using Magify.Types;
using JetBrains.Annotations;
using UnityEngine;
using EventType = Magify.Model.EventType;

namespace Magify
{
    internal class AnalyticsService : IDisposable
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Analytics);
        [NotNull]
        private readonly PooledCompositeDisposable _disposables = new();

        [NotNull]
        private readonly AnalyticsTracker _analyticsTracker;
        [NotNull]
        private readonly CampaignsTracker _campaignsTracker;

        public AnalyticsService([NotNull] AnalyticsTracker analyticsTracker, [NotNull] CampaignsTracker campaignsTracker)
        {
            _analyticsTracker = analyticsTracker;
            _campaignsTracker = campaignsTracker;
        }

        internal void SetGameLevel(int level)
        {
            _analyticsTracker.LevelState.Level = level;
        }

        internal void SetGameMaxLevel(int maxLevel)
        {
            _analyticsTracker.LevelState.MaxLevel = maxLevel;
        }

        internal void SetGameMode([CanBeNull] string mode)
        {
            _analyticsTracker.LevelState.Mode = mode;
        }

        public void SetupAnalyticsConfig(int eventsGroupSize, int syncTimeInterval)
        {
            var analyticsConfiguration = new AnalyticsConfiguration
            {
                GroupSize = eventsGroupSize,
                SyncIntervalInSeconds = syncTimeInterval
            };
            _logger.Log($"Setup analytics config: ${analyticsConfiguration}");
            _analyticsTracker.SetupAnalyticsConfig(analyticsConfiguration);
        }

        public void ResetAnalyticsConfig()
        {
            _logger.Log("Reset analytics config");
            _analyticsTracker.ResetAnalyticsConfig();
        }

        public void TrackCampaignImpression([CanBeNull] string campaignName, bool includeNested)
        {
            var request = _campaignsTracker.GetCampaignRequest(campaignName);
            var impression = _campaignsTracker.CreateCampaignImpression(campaignName);
            if (request == null || impression == null)
            {
                _logger.LogError($"Can't track campaign impression event. Campaign record was not found: {campaignName}");
                return;
            }

            _analyticsTracker.TrackCommonEvent(EventType.Impression, impression);
            if (includeNested && request.Campaign.IsSupportNested())
            {
                TrackNestedCampaignImpressions(campaignName, request.NestedCampaigns!.Select(p => p?.ProductId).ToList());
            }
        }

        public void TrackNestedCampaignImpressions([CanBeNull] string campaignName, IReadOnlyList<string> productIds)
        {
            var impressions = _campaignsTracker.GetNestedCampaignImpressions(campaignName, productIds);
            if (impressions.Count > 0)
            {
                _analyticsTracker.TrackCommonEvents(EventType.Impression, impressions);
            }
        }

        public void TrackImpressionFailEvent([CanBeNull] string campaignName, [CanBeNull] string reason)
        {
            var impression = _campaignsTracker.CreateCampaignImpression(campaignName);
            if (impression == null)
            {
                _logger.LogError($"Can't track impression fail event. Campaign record was not found: {campaignName}");
                return;
            }

            _analyticsTracker.TrackImpressionFailEvent(impression, reason);
        }

        public void TrackClickEvent(EventType eventType, [CanBeNull] string campaignName, [CanBeNull] string productId = null, [CanBeNull] PurchaseStore? store = null)
        {
            var request = _campaignsTracker.GetCampaignRequest(campaignName);
            var impression = _campaignsTracker.CreateCampaignImpression(campaignName);
            if (request == null || impression == null)
            {
                _logger.LogError($"Can't track click event: {eventType}. Campaign record was not found: {campaignName}");
                return;
            }

            if (productId != null && request.TryGetNestedByProduct(productId, store, out var nestedCampaign))
            {
                impression.CampaignName = nestedCampaign.Name;
                impression.ProductId = nestedCampaign.ProductId;
                impression.ProductIdCreative = nestedCampaign.ProductIdCreative;
                impression.StoreName = nestedCampaign.StoreName;
            }

            _analyticsTracker.TrackCommonEvent(eventType, impression);
        }

        public void TrackAdsImpression([CanBeNull] string campaignName, [NotNull] IAdsImpression adsImpression)
        {
            var request = _campaignsTracker.GetCampaignRequest(campaignName);
            var impression = _campaignsTracker.CreateCampaignImpression(campaignName);
            if (request == null || impression == null)
            {
                _logger.LogError($"Can't track ads impression event. Campaign record was not found: {campaignName}");
                return;
            }
            if (request.Campaign.Type.HasNestedAdsCampaigns())
            {
                TrackNestedAdsImpression(request, adsImpression);
            }
            else
            {
                TrackAdsImpressionEvent(impression, adsImpression);
            }
        }

        private void TrackNestedAdsImpression([NotNull] CampaignRequest request, [NotNull] IAdsImpression adsImpression)
        {
            var impression = _campaignsTracker.GetNestedAdsCampaignImpression(request.Campaign.Name);
            if (impression == null)
            {
                _logger.LogError($"Can't track nested ads impression event. Campaign record was not found: {request.Campaign.Name}");
                return;
            }
            TrackAdsImpressionEvent(impression, adsImpression);
        }

        private void TrackAdsImpressionEvent([NotNull] CampaignImpression impression, [NotNull] IAdsImpression adsImpression)
        {
            switch (adsImpression)
            {
                case ApplovinAdsImpression applovinData:
                    _analyticsTracker.TrackApplovinAdsImpression(impression, applovinData);
                    break;
                case IronSourceAdsImpression ironSourceData:
                    _analyticsTracker.TrackIronSourceAdsImpression(impression, ironSourceData);
                    break;
                default:
                    _logger.LogError($"Can't track ads impression event. Unknown impression data: {JsonFacade.SerializeObject(adsImpression)}");
                    break;
            }
        }

        public void TrackSubscription(bool isTrial, string productId, string price, string currency, [CanBeNull] string transactionId, [CanBeNull] string originalTransactionId, PurchaseStore store, [CanBeNull] Option<string> customStoreFront)
        {
            var eventType = isTrial ? EventType.TrialActivation : EventType.PaidSubscriptionActivation;
            var product = new ProductInfo(productId, price, currency, store);
            var purchase = new PurchaseInfo
            {
                TransactionId = transactionId,
                OriginalTransactionId = originalTransactionId,
                Product = product,
                CustomStoreFront = customStoreFront,
            };
            TrackPurchaseEvent(eventType, purchase);
        }

        public void TrackInApp(bool isExternal, string productId, string price, string currency, [CanBeNull] string transactionId, [CanBeNull] string originalTransactionId, PurchaseStore store, [CanBeNull] Option<string> customStoreFront)
        {
            var product = new ProductInfo(productId, price, currency, store);
            var purchase = new PurchaseInfo
            {
                TransactionId = transactionId,
                OriginalTransactionId = originalTransactionId,
                Product = product,
                CustomStoreFront = customStoreFront,
            };
            if (isExternal)
            {
                _analyticsTracker.TrackPurchaseEvent(EventType.InApp, _campaignsTracker.BuildExternalInAppImpression(productId), purchase);
            }
            else
            {
                TrackPurchaseEvent(EventType.InApp, purchase);
            }
        }

        private void TrackPurchaseEvent(EventType type, PurchaseInfo purchase)
        {
            var impression = _campaignsTracker.GetNestedCampaignImpression(purchase.Product.ProductId, purchase.Product.Store);
            if (impression == null)
            {
                _logger.LogError($"Can't track purchase event: {type}. Campaign record was not found for product: {purchase.Product.ProductId}");
                return;
            }

            _analyticsTracker.TrackPurchaseEvent(type, impression, purchase);
        }

        public void TrackIncomeTransaction(string source, [NotNull] List<BonusInfo> bonuses, [CanBeNull] ProductInfo product)
        {
            if (bonuses.Count == 0)
            {
                _logger.LogError("Invalid income transaction. Bonus info can't be empty.");
                return;
            }

            var impression = product?.ProductId != null
                ? _campaignsTracker.GetNestedCampaignImpression(product.ProductId, product.Store)
                : null;
            _analyticsTracker.TrackTransaction(Transaction.CreateIncome(source), bonuses, product, impression);
        }

        internal void TrackExpenseTransaction([NotNull] List<BonusInfo> bonuses)
        {
            if (bonuses.Count == 0)
            {
                _logger.LogError("Invalid expense transaction. Bonus info can't be empty.");
                return;
            }

            _analyticsTracker.TrackTransaction(Transaction.CreateExpense(), bonuses);
        }

        internal void TrackCorrectionTransaction([NotNull] List<BonusInfo> bonuses)
        {
            if (bonuses.Count == 0)
            {
                _logger.LogError("Invalid correction transaction. Bonus info can't be empty.");
                return;
            }

            _analyticsTracker.TrackTransaction(Transaction.CreateCorrection(), bonuses);
        }

        public void TrackCustomSessionEvent(string eventName, Dictionary<string, object> parameters)
        {
            _analyticsTracker.TrackCustomSessionEvent(eventName, parameters);
        }

        public void Dispose()
        {
            _disposables.Release();
        }
    }
}