﻿using System;
using System.Collections.Generic;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Rx;
using UnityEngine;

namespace Magify.Aghanim
{
    internal class AghanimManager : IAghanimApi, IInitializable, ICancelable
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Aghanim);

        [NotNull]
        private readonly IAghanimServerApi _serverApi;
        [NotNull]
        private readonly AghanimPrefs _prefs;
        [NotNull]
        private readonly PooledCompositeDisposable _disposables = new();

        [NotNull, ItemNotNull]
        private readonly HashSet<string> _activePendingOrders = new(16);
        [NotNull]
        private readonly Dictionary<string, UniTaskCompletionSource<AghanimOrderStatus>> _finalOrderStatusAwaiters = new();
        [NotNull]
        private readonly Subject<(string OrderId, AghanimOrderStatus Status)> _onOrderStatusChanged = new();
        [NotNull]
        private readonly SyncTimer _syncTimer;

        [NotNull]
        public IObservable<(string OrderId, AghanimOrderStatus Status)> OnOrderStatusChanged => _onOrderStatusChanged;
        [NotNull]
        public IReactiveProperty<float> VerificationRetryInterval { get; } = new ReactiveProperty<float>(4.0f);
        private int RetryIntervalMills => Mathf.Max(Mathf.RoundToInt(VerificationRetryInterval.Value * 1000), 1000);
        public bool IsDisposed => _disposables.IsDisposed;

        public AghanimManager(
            [NotNull] IAghanimServerApi serverApi,
            [NotNull] AghanimPrefs prefs)
        {
            _serverApi = serverApi;
            _prefs = prefs;
            _syncTimer = new SyncTimer(GetProductStatuses).AddTo(_disposables)!;
        }

        public void Dispose()
        {
            _disposables.Release();
        }

        void IInitializable.Initialize()
        {
            GetProductStatuses();
            _syncTimer.Start(RetryIntervalMills);
        }

        public async UniTask<AghanimOrder> IssueOrder([NotNull] string productId, bool isSandbox, CancellationToken cancellationToken)
        {
            foreach (var (storedOrderId, storedProductId) in _prefs.OrderToProductMap)
            {
                if (storedOrderId != null
                 && storedProductId == productId
                 && _prefs.TryGetOrderWithStatus(storedOrderId, out var storedOrderStatus)
                 && storedOrderStatus is AghanimOrderStatus.Pending
                 && _prefs.TryGetOrderInfo(storedOrderId, out var currency, out var price)
                 && _prefs.OrderToUrlMap.TryGetValue(storedOrderId, out var url)
                 && !string.IsNullOrEmpty(url))
                {
                    return new AghanimOrder
                    {
                        Id = storedOrderId,
                        Url = url,
                        Price = price,
                        Currency = currency
                    };
                }
            }

            var order = await _serverApi.CreateOrder(productId, isSandbox, cancellationToken);
            if (order != null)
            {
                using (_prefs.MultipleChangeScope())
                {
                    _prefs.SetOrderStatus(order.Id, AghanimOrderStatus.Pending);
                    _prefs.SaveOrderInfo(order);
                    _prefs.OrderToProductMap.Add(order.Id, productId);
                    _prefs.OrderToUrlMap.Add(order.Id, order.Url);
                }
            }
            return order;
        }

        [NotNull]
        public IEnumerable<AghanimOrderInfo> IterateAllOrders()
        {
            foreach (var orderId in _prefs.Orders)
            {
                if (orderId != null)
                {
                    var status = _prefs.GetOrderStatus(orderId);
                    _prefs.GetOrderInfo(orderId, out var price, out var currency);
                    yield return new AghanimOrderInfo(orderId, status, price, currency);
                }
            }
        }

        public UniTask<IReadOnlyList<AghanimProductCounting>> LoadPurchasedProductsAsync(CancellationToken cancellationToken)
        {
            return _serverApi.GetProductsCount(cancellationToken);
        }

        [ContractAnnotation("=> true, productId:notnull; => false, productId:null")]
        public bool TryGetProductOfOrder([NotNull] string orderId, [CanBeNull] out string productId)
        {
            return _prefs.OrderToProductMap.TryGetValue(orderId, out productId);
        }

        public bool TryGetOrderInfo([NotNull] string orderId, out AghanimOrderInfo info)
        {
            if (_prefs.TryGetOrderInfo(orderId, out var currency, out var price)
             && _prefs.TryGetOrderWithStatus(orderId, out var status))
            {
                info = new AghanimOrderInfo(orderId, status, price, currency);
                return true;
            }
            info = default(AghanimOrderInfo);
            return false;
        }

        public bool TryGetOrderWithStatus([NotNull] string orderId, out AghanimOrderStatus status)
        {
            return _prefs.TryGetOrderWithStatus(orderId, out status);
        }

        public UniTask<AghanimOrderStatus> WaitForFinalOrderStatusAsync([NotNull] string orderId, CancellationToken cancellationToken)
        {
            if (_prefs.TryGetOrderWithStatus(orderId, out var status) && status.IsFinal())
            {
                return UniTask.FromResult(status);
            }
            UniTaskCompletionSource<AghanimOrderStatus> promise;
            lock (_finalOrderStatusAwaiters)
            {
                if (!_finalOrderStatusAwaiters.TryGetValue(orderId, out promise) || promise == null)
                {
                    _finalOrderStatusAwaiters[orderId] = promise = new UniTaskCompletionSource<AghanimOrderStatus>();
                }
            }
            GetOrderStatusAsync(orderId, _disposables.GetOrCreateToken()).Forget();
            _syncTimer.Restart();
            return promise.Task.AttachExternalCancellation(cancellationToken);
        }

        private void GetProductStatuses()
        {
            foreach (var orderId in _prefs.Orders)
            {
                if (orderId == null) continue;
                if (_prefs.GetOrderStatus(orderId) is AghanimOrderStatus.Pending)
                {
                    GetOrderStatusAsync(orderId, _disposables.GetOrCreateToken()).Forget();
                }
            }
        }

        private async UniTaskVoid GetOrderStatusAsync([NotNull] string orderId, CancellationToken cancellationToken)
        {
            if (_serverApi.IsDisposed)
                return;
            lock (_activePendingOrders)
                if (!_activePendingOrders.Add(orderId))
                    return;

            var retryImmediately = false;
            try
            {
                await TaskScheduler.SwitchToThreadPool(cancellationToken);
                var result = await _serverApi.GetOrderStatus(orderId, cancellationToken);
                await TaskScheduler.SwitchToMainThread(cancellationToken);
                switch (result)
                {
                    case { } status when status.IsFinal():
                        using (_prefs.MultipleChangeScope())
                        {
                            _prefs.SetOrderStatus(orderId, status);
                            _prefs.OrderToUrlMap.Remove(orderId);
                        }
                        lock (_finalOrderStatusAwaiters)
                        {
                            if (_finalOrderStatusAwaiters.TryGetValue(orderId, out var promise))
                            {
                                try
                                {
                                    promise?.TrySetResult(status);
                                }
                                catch (Exception e)
                                {
                                    _logger.LogError($"An exception occured (and ignored) while notifying order status changed: {e}");
                                }
                            }
                        }
                        try
                        {
                            _onOrderStatusChanged.OnNext((orderId, status));
                        }
                        catch (Exception e)
                        {
                            _logger.LogError($"An exception occured (and ignored) while notifying order status changed: {e}");
                        }
                        break;
                }
            }
            catch (OperationCanceledException)
            {
                _logger.Log($"Getting order status request for order '{orderId}' was cancelled, retrying immediately");
                retryImmediately = true; // we can set `true` here because below we check `IsDisposed` and only then call `_disposables.GetOrCreateToken()`
            }
            finally
            {
                if (!IsDisposed)
                {
                    lock (_activePendingOrders)
                        _activePendingOrders.Remove(orderId);
                    if (retryImmediately)
                    {
                        _logger.Log($"Immediately retrying to get order status for order '{orderId}'");
                        GetOrderStatusAsync(orderId, _disposables.GetOrCreateToken()).Forget();
                    }
                    else
                    {
                        _syncTimer.Restart();
                    }
                }
            }
        }
    }
}