using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using UnityEngine;
using UnityEngine.Networking;

namespace Magify
{
    public abstract class RemoteStorage<TContent>
        where TContent : class
    {
        private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Storage);

        [NotNull]
        private readonly Dictionary<string, DownloadPromise<StorageResultCode>> _loaders = new();
        [NotNull]
        private readonly CacheContent<TContent> _cache;
        [NotNull]
        private readonly IRemoteStorageNetworkClient _networkClient;
        private bool _isApplicationQuitting;
        [NotNull]
        private readonly object _lock = new();

        protected string RootPath { get; }

        protected RemoteStorage(string storagePath, string rootFolder) : this(storagePath, rootFolder, new RemoteStorageNetworkClient())
        {
        }

        protected RemoteStorage(string storagePath, string rootFolder, [NotNull] IRemoteStorageNetworkClient networkClient, bool clearCache = true)
        {
            _networkClient = networkClient;
            RootPath = $"{storagePath}/{rootFolder}";
            Application.quitting += OnApplicationQuit;
            _cache = new CacheContent<TContent>(ReleaseContentInternal);

            try
            {
                CreateRootFolder();
                if (clearCache)
                    ClearCache();
            }
            catch
            {
                // ignored
            }
        }

        internal virtual async UniTask<ContentHandle<TContent>> Load([NotNull] string url, double timeout, CancellationToken cancellationToken)
        {
            await TaskScheduler.SwitchToMainThread(cancellationToken);
            LogWithPath("Begin download", url);
            if (HasActiveDownloadPromise(url, out _))
            {
                return await TryLoadFromRemote(url, timeout, cancellationToken);
            }
            return TryLoadFromDisk(url) ?? await TryLoadFromRemote(url, timeout, cancellationToken);
        }

        [CanBeNull]
        protected ContentHandle<TContent> TryLoadFromDisk([NotNull] string url)
        {
            var content = GetFromCacheOrDisk(url);
            if (content != null)
            {
                return CreateContentHandle(url, StorageResultCode.Success, content);
            }
            if (!HasInternetConnection())
            {
                LogWithPath("Can't begin downloading without internet connection", url);
                return CreateContentHandle(url, StorageResultCode.ConnectionError);
            }
            return null;
        }

        protected async UniTask<ContentHandle<TContent>> TryLoadFromRemote([NotNull] string url, double timeout, CancellationToken cancellationToken)
        {
            // When downloadPromiseHandle is disposed, the number of waiters of downloading from this 'url' will be decreased.
            // When number of waiters is 0, the download operation will be cancelled if it's not completed.
            using var downloadPromiseHandle = GetDownloadPromise(url);
            var timeoutSpan = timeout <= 0 ? TimeSpan.FromMinutes(10) : TimeSpan.FromSeconds(timeout);
            var delayTask = UniTask.Delay(timeoutSpan, cancellationToken: cancellationToken);

            try
            {
                cancellationToken.ThrowIfCancellationRequested();
                var (hasDownloadResult, resultCode) = await UniTask.WhenAny(downloadPromiseHandle.Task, delayTask);
                cancellationToken.ThrowIfCancellationRequested();
                switch (hasDownloadResult)
                {
                    case false:
                        LogWithPath("Download timeout", url);
                        return CreateContentHandle(url, StorageResultCode.Timeout);
                    case true when resultCode == StorageResultCode.Success:
                        LogWithPath("Successfully downloaded", url);
                        return CreateContentHandle(url, StorageResultCode.Success, _cache.Get(url));
                    case true when resultCode == StorageResultCode.AlreadyDownloaded:
                        LogWithPath("Actual content already downloaded and might be loaded from disk", url);
                        return CreateContentHandle(url, StorageResultCode.AlreadyDownloaded);
                    default:
                        LogWithPath($"Failed to download with error {resultCode}", url);
                        return CreateContentHandle(url, resultCode);
                }
            }
            catch (OperationCanceledException)
            {
                LogWithPath("Operation was canceled", url);
                return CreateContentHandle(url, StorageResultCode.Canceled);
            }
        }

        internal string GetCachePath(string url)
        {
            string fileName;
#if UNITY_EDITOR
            var extension = Path.GetExtension(url);
            if (extension == ".bundle")
            {
                fileName = $"{Hash128.Compute(url)}{extension}";
            }
            else
#endif
            {
                fileName = Path.GetFileName(url);
            }

            return Path.Combine(RootPath, fileName);
        }

        /// <summary>
        /// Creates a new download operation handle or returns existing one. <br/>
        /// When operation is completed all DownloadPromise.Handle.<see cref="DownloadPromise.Handle.Task"/> will be completed. <br/>
        /// When all handles are disposed, but download operation is Not completed, the operation will be cancelled and removed from the dictionary. <br/>
        /// </summary>
        private DownloadPromise<StorageResultCode>.Handle GetDownloadPromise([NotNull] string url)
        {
            if(HasActiveDownloadPromise(url, out var promise))
            {
                LogWithPath("There is existing operation for content loading", url);
            }
            else
            {
                LogWithPath("Start new operation for content downloading", url);
                promise = new DownloadPromise<StorageResultCode>();
                lock (_lock)
                {
                    _loaders[url] = promise;
                }
                download(promise.OnDisposedCancellationToken).Forget();
            }
            return promise.GetHandle();

            async UniTaskVoid download(CancellationToken cancellationToken)
            {
                try
                {
                    // Doesn't have any download timeout.
                    var result = await DownloadFromUrlInCache(url, cancellationToken);
                    promise.TrySetResult(result);
                }
                catch (OperationCanceledException)
                {
                    LogWithPath($"{nameof(DownloadFromUrlInCache)} was cancelled", url);
                }
                finally
                {
                    lock (_lock)
                    {
                        _loaders.Remove(url);
                    }
                }
            }
        }

        [ContractAnnotation("=> true, promise:notnull; => false, promise:null")]
        private bool HasActiveDownloadPromise([NotNull] string url, out DownloadPromise<StorageResultCode> promise)
        {
            lock (_lock)
            {
                return _loaders.TryGetValue(url, out promise) && promise?.IsDisposed is false;
            }
        }

        /// <remarks>
        /// Doesn't have any download timeout.
        /// </remarks>
        internal async UniTask<StorageResultCode> DownloadFromUrlInCache([NotNull] string url, CancellationToken cancellationToken)
        {
            var result = await DownloadFromUrlWithoutTimeout(url, cancellationToken);
            if (result.Code == StorageResultCode.Success)
            {
                if (!_cache.TryStore(url, result.Value))
                {
                    _logger.LogWarning("A resource is trying to be cached, but there is some content handle of the same resource that has not been disposed of." +
                                       $"You need to dispose of the content handle to replace it with a new one. Resource url: {url}");
                }
            }
            return result.Code;
        }

        internal bool IsStoredInCache(string url)
        {
            return _cache.Contains(url);
        }

        internal bool IsCachedOnDisk(string url)
        {
            return File.Exists(GetCachePath(url));
        }

        internal void SaveToDisk(string url, byte[] bytes)
        {
            var filePath = PathForSavingToDisk(url);
            var filePathTemp = filePath + ".tmp";
            try
            {
                if (File.Exists(filePathTemp))
                {
                    LogWithPath($"Delete temporary file {filePathTemp}", url);
                    File.Delete(filePathTemp);
                }
                LogWithPath("Save file on disk", url);
                File.WriteAllBytes(filePathTemp, bytes);
                if (File.Exists(filePath))
                {
                    LogWithPath($"Delete existing file {filePath}", url);
                    File.Delete(filePath);
                }
                File.Move(filePathTemp, filePath);
            }
            catch (Exception e)
            {
                LogErrorWithPath("Failed to save on disk at " + filePath.Replace("/", "\\"), url);
                _logger.LogException(e);
            }
        }

        internal string PathForSavingToDisk(string url)
        {
            CreateRootFolder();
            return GetCachePath(url);
        }

        protected async UniTask<StorageResultCode> SendWebRequest([NotNull] UnityWebRequest www, CancellationToken cancellationToken)
        {
            return await _networkClient.SendWebRequest(www, cancellationToken);
        }

        private TContent GetFromCacheOrDisk(string url)
        {
            var content = GetFromCache(url);
            if (content != null)
            {
                LogWithPath("Loaded from cache", url);
                return content;
            }

            content = GetFromDisk(url);
            if (content != null)
            {
                LogWithPath("Loaded from disk", url);
                return content;
            }

            return null;
        }

        private TContent GetFromCache(string url)
        {
            LogWithPath($"Start {nameof(GetFromCache)}", url);
            if (!IsStoredInCache(url))
            {
                LogWithPath("Content is not in cache", url);
                return null;
            }

            LogWithPath($"Get content from cache", url);
            var content = _cache.Get(url);
            return content;
        }

        private TContent GetFromDisk(string url)
        {
            if (!IsCachedOnDisk(url))
            {
                LogWithPath("Content is not cached on disk", url);
                return null;
            }
            try
            {
                var content = LoadFromDiskInCache(url);
                if (content != null)
                {
                    return content;
                }
            }
            catch (Exception e)
            {
                LogErrorWithPath($"Failed to load from disk. See details below", url);
                _logger.LogException(e);
            }

            return null;
        }

        internal TContent LoadFromDiskInCache(string url)
        {
            var filePath = GetCachePath(url);
            var content = LoadFromDisk(filePath);
            return _cache.StoreAndGet(url, content);
        }

        protected virtual bool HasInternetConnection()
        {
            return Application.internetReachability != NetworkReachability.NotReachable;
        }

        protected abstract TContent LoadFromDisk(string url);
        protected abstract UniTask<(StorageResultCode Code, TContent Value)> DownloadFromUrlWithoutTimeout([NotNull] string url, CancellationToken cancellationToken);
        protected abstract void ReleaseContent(TContent cache);

        private void ReleaseContentInternal(TContent cache)
        {
            if (_isApplicationQuitting) return;
            ReleaseContent(cache);
        }

        private ContentHandle<TContent> CreateContentHandle(string url, StorageResultCode code, TContent value = null)
        {
            Action callback = null;
            if (value != null)
            {
                callback = () => _cache.Release(url);
            }
            return new ContentHandle<TContent>(code, value, callback);
        }

        private ContentHandle<TContent> CreateContentHandle(string url, (StorageResultCode Code, TContent Value) data)
        {
            return CreateContentHandle(url, data.Code, data.Value);
        }

        protected void ClearCache(bool force = false)
        {
            _logger.Log("Clearing cache from old files");
            var files = Directory.GetFiles(RootPath);
            foreach (var file in files)
            {
                var fi = new FileInfo(file);
                _logger.Log($"{file}: LastAccessTime: {fi.LastAccessTime} ({(DateTime.Now - fi.LastAccessTime).TotalDays:0.##} days old)");
                if (force || fi.LastAccessTime < DateTime.Now.AddDays(-7))
                {
                    if (PrepareForCacheFileDeletion(fi))
                    {
                        _logger.Log($"Delete {file}");
                        fi.Delete();
                    }
                }
                else
                {
                    _logger.Log($"Keep {file}");
                }
            }
        }

        protected virtual bool PrepareForCacheFileDeletion([NotNull] FileInfo fileInfo)
        {
            return true;
        }

        private void CreateRootFolder()
        {
            if (!Directory.Exists(RootPath))
            {
                Directory.CreateDirectory(RootPath);
            }
        }

        private void OnApplicationQuit()
        {
            _isApplicationQuitting = true;
        }

        [Conditional(MagifyLogger.VerboseLoggingDefine)]
        protected void LogWithPath(string message, string url)
        {
            _logger.Log($"{message}\n{Path.GetFileName(url)}: {url}");
        }

        protected void LogErrorWithPath(string message, string url)
        {
            _logger.LogError($"{message}\n{Path.GetFileName(url)}: {url}");
        }
    }
}