﻿#if false
using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.InteropServices;
using Newtonsoft.Json;
using UnityEngine;

namespace Magify
{
    internal partial class MagifyiOS : MagifyPlatformAPI, ILegacyBridgePlatformAPI
    {
        private static event Action OnFeaturesUpdatedFromNative;

        internal override event Action OnConfigLoaded;
        private static event Action OnConfigLoadedFromNative;

        private static event Action<LtoInfo> OnLtoDidFinishFromNative;
        private static event Action<IReadOnlyCollection<LtoInfo>> OnLtoDidUpdateFromNative;

        public event Action<LtoInfo> OnLtoDidFinish;
        public event Action<IReadOnlyCollection<LtoInfo>> OnLtoDidUpdate;

        private static Dictionary<string, Action<ICampaign>> CampaginUpdateCallbacks = new Dictionary<string, Action<ICampaign>>();

        private static event Action<int> OnSessionChangedFromNative;
        internal override event Action<int> OnSessionChanged;

        private static event Action OnSubscriptionStatusChangedFromNative;
        internal override event Action OnSubscriptionStatusChanged;
        private static event Action OnInAppStatusChangedFromNative;
        internal override event Action OnInAppStatusChanged;
        private static event Action OnAuthorizationStatusChangedFromNative;
        internal override event Action OnAuthorizationStatusChanged;
        private static event Action OnReferrerChangedFromNative;
        internal override event Action OnReferrerChanged;
        private static event Action OnPurchasedProductsChangedFromNative;
        internal override event Action OnPurchasedProductsChanged;

        private static event Action OnApplicationEnterForegroundFromNative;
        internal override event Action OnApplicationEnterForeground;
        private static event Action OnApplicationEnterBackgroundFromNative;
        internal override event Action OnApplicationEnterBackground;

        private static event Action OnUserDidTakeScreenshotFromNative;
        internal override event Action OnUserDidTakeScreenshot;

        private readonly JsonSerializerSettings _jsonSerializerSettings = new()
        {
            NullValueHandling = NullValueHandling.Ignore,
        };
        private static readonly MagifyLogger _logger = MagifyLogger.Get();
        private readonly NativeFeaturesProvider _featuresProvider;

        internal override IReadOnlyCollection<LtoInfo> ActiveLtoOffers
        {
            get
            {
                var pointer = _magifyGetActiveLtoOffers();
                var ltoSetString = (string)Marshal.PtrToStringAnsi(pointer);
                var activeLtos = LtoInfosFromJSON(ltoSetString);
                Marshal.FreeHGlobal(pointer);

                return activeLtos;
            }
        }

        internal override Environment Environment
        {
            get => (Environment) _magifyGetEnvironment();
            set => _magifySetEnvironment((int) value);
        }

        internal override SubscriptionStatus SubscriptionStatus
        {
            get => (SubscriptionStatus) _magifyGetSubscriptionStatus();
            set => _magifySetSubscriptionStatus((int) value);
        }

        internal override bool IsLoggingEnabled
        {
            get => _magifyGetIsLoggingEnabled();
            set => _magifySetIsLoggingEnabled(value);
        }

        internal override string ReferrerId
        {
            get => _magifyGetReferrerId();
            set => _magifySetReferrerId(value);
        }

        internal override string AdjustId
        {
            get => _magifyGetAdjustId();
            set => _magifySetAdjustId(value);
        }

        internal override string FirebaseInstanceId
        {
            get => _magifyGetFirebaseInstanceId();
            set => _magifySetFirebaseInstanceId(value);
        }

        internal override bool IsPortrait => _magifyIsPortrait();

        internal override IFeatures Features => _featuresProvider;
        internal override Limits Limits => _magifyGetLimits();

        internal override string ClientId => _magifyGetClientId();

        internal override int SessionNumber => _magifySessionNumber();

        internal override AuthorizationStatus AuthorizationStatus
        {
            get => (AuthorizationStatus) _magifyGetAuthorizationStatus();
            set => _magifySetAuthorizationStatus((int) value);
        }

        internal override InAppStatus InAppStatus
        {
            get => (InAppStatus)_magifyGetInAppStatus();
            set => throw new NotImplementedException();
        }

        internal override ContextSyncTime LastContextSyncTime
        {
            get
            {
                var syncTimePtr = _magifyGetContextSyncTime();
                var syncTimeModel = (ContextSyncTimeModel)Marshal.PtrToStructure(syncTimePtr, typeof(ContextSyncTimeModel));
                return new ContextSyncTime
                {
                    LocalTimeMillis = (long)(syncTimeModel.LocalTimeInterval * 1000),
                    RemoteTimeMillis = (long)(syncTimeModel.RemoteTimeInterval * 1000)
                };
            }
        }

        internal override List<InAppProduct> InAppProducts
        {
            get
            {
                var productsString = _magifyInappProducts();
                return ParseInAppProductsList(productsString);
            }
        }

        internal override List<SubscriptionProduct> SubscriptionProducts
        {
            get
            {
                var productsString = _magifySubscriptionProducts();
                return ParseSubscriptionProductsList(productsString);
            }
        }

        internal string DefaultFeatures => _magifyGetDefaultFeatures();
        internal string CurrentFeatures => _magifyGetCurrentFeatures();

        internal override bool IsSdkInitialized => _magifyIsSdkInitialized();


        public MagifyiOS()
        {
            _featuresProvider = new NativeFeaturesProvider();
        }

        internal override void InitializeSdk(string appName, string configPath, string clientId, Environment environment)
        {
            _magifyInitialize(appName, configPath, clientId);
            OnLtoDidFinishFromNative += (info) => { OnLtoDidFinish?.Invoke(info); };
            OnLtoDidUpdateFromNative += (infoSet) => { OnLtoDidUpdate?.Invoke(infoSet); };
            _magifySetupLTOCallback(LtoDidFinishHandler, LtoDidUpdateHandler);

            OnFeaturesUpdatedFromNative += () => { _featuresProvider.UpdateCurrentFeatures(CurrentFeatures); };
            OnConfigLoadedFromNative += () => { OnConfigLoaded?.Invoke(); };
            _magifySetupFeatures(FeaturesUpdatedFromIOSPlatform);
            _magifySetupOnCampaignsLoaded(CampaignsLoadedFromIOSPlatform);

            OnSessionChangedFromNative += (sessionNumber) => { OnSessionChanged?.Invoke(sessionNumber); };
            _magifySetSessionNumberChanged(SessionChangedHandler);

            OnSubscriptionStatusChangedFromNative += () => { OnSubscriptionStatusChanged?.Invoke(); };
            _magifySetSubscriptionStatusChanged(SubscriptionStatusChangedHandler);

            OnInAppStatusChangedFromNative += () => { OnInAppStatusChanged?.Invoke(); };
            _magifySetInAppStatusChanged(InAppStatusChangedHandler);

            OnAuthorizationStatusChangedFromNative += () => { OnAuthorizationStatusChanged?.Invoke(); };
            _magifySetAuthorizationStatusChanged(AuthorizationStatusChangedHandler);

            OnReferrerChangedFromNative += () => { OnReferrerChanged?.Invoke(); };
            _magifySetReferrerIdChanged(ReferrerChangedHandler);

            OnPurchasedProductsChangedFromNative += () => { OnPurchasedProductsChanged?.Invoke(); };
            _magifySetPurchasedProductsChanged(PurchasedProductsChangedHandler);

            OnApplicationEnterForegroundFromNative += () => { OnApplicationEnterForeground?.Invoke(); };
            _magifySetApplicationEnterForegroundCallback(OnApplicationEnterForegroundHandler);

            OnApplicationEnterBackgroundFromNative += () => { OnApplicationEnterBackground?.Invoke(); };
            _magifySetApplicationEnterBackgroundCallback(OnApplicationEnterBackgroundHandler);

            OnUserDidTakeScreenshotFromNative += () => { OnUserDidTakeScreenshot?.Invoke(); };
            _magifySetUserDidTakeScreenshotCallback(OnUserDidTakeScreenshotHandler);
        }

        [AOT.MonoPInvokeCallback(typeof(Action<IntPtr>))]
        private static void LtoDidFinishHandler(IntPtr pointer)
        {
            var ltoInfoModel = (LtoInfoModel)Marshal.PtrToStructure(pointer, typeof(LtoInfoModel));
            var ltoInfo = new LtoInfo
            {
                CampaignName = ltoInfoModel.CampaignName,
                Spot = ltoInfoModel.Spot,
                StartTime = UnixTimestamp.ToDateTime(ltoInfoModel.StartTime),
                EndTime = UnixTimestamp.ToDateTime(ltoInfoModel.EndTime),
                BadgeCreative = new LtoBadgeCreative
                {
                    Tag = ltoInfoModel.BadgeCreativeModel.Tag,
                    Title = ltoInfoModel.BadgeCreativeModel.Title,
                    LabelText = ltoInfoModel.BadgeCreativeModel.LabelText,
                    Resource = ParseCreativeResource(ltoInfoModel.BadgeCreativeModel.Resource),
                    Placeholder = ltoInfoModel.BadgeCreativeModel.Placeholder,
                    HasTimer = ltoInfoModel.BadgeCreativeModel.HasTimer,
                    LastMinutesTimer = ltoInfoModel.BadgeCreativeModel.LastMinutesTimer,
                    Attributes = Json.Deserialize(ltoInfoModel.BadgeCreativeModel.Attributes) as Dictionary<string, object>
                }
            };

            _magifyFreeLtoInfo(pointer);
            OnLtoDidFinishFromNative?.Invoke(ltoInfo);
        }

        [AOT.MonoPInvokeCallback(typeof(Action<IntPtr>))]
        private static void LtoDidUpdateHandler(IntPtr pointer)
        {
            var ltoSetString = (string)Marshal.PtrToStringAnsi(pointer);
            var activeLto = LtoInfosFromJSON(ltoSetString);
            Marshal.FreeHGlobal(pointer);

            OnLtoDidUpdateFromNative?.Invoke(activeLto);
        }

        [AOT.MonoPInvokeCallback(typeof(Action<int>))]
        private static void SessionChangedHandler(int sessionNumber)
        {
            OnSessionChangedFromNative?.Invoke(sessionNumber);
        }

        [AOT.MonoPInvokeCallback(typeof(Action))]
        private static void SubscriptionStatusChangedHandler()
        {
            OnSubscriptionStatusChangedFromNative?.Invoke();
        }

        [AOT.MonoPInvokeCallback(typeof(Action))]
        private static void InAppStatusChangedHandler()
        {
            OnInAppStatusChangedFromNative?.Invoke();
        }

        [AOT.MonoPInvokeCallback(typeof(Action))]
        private static void AuthorizationStatusChangedHandler()
        {
            OnAuthorizationStatusChangedFromNative?.Invoke();
        }

        [AOT.MonoPInvokeCallback(typeof(Action))]
        private static void ReferrerChangedHandler()
        {
            OnReferrerChangedFromNative?.Invoke();
        }

        [AOT.MonoPInvokeCallback(typeof(Action))]
        private static void PurchasedProductsChangedHandler()
        {
            OnPurchasedProductsChangedFromNative?.Invoke();
        }

        [AOT.MonoPInvokeCallback(typeof(Action))]
        private static void OnApplicationEnterForegroundHandler()
        {
            OnApplicationEnterForegroundFromNative?.Invoke();
        }

        [AOT.MonoPInvokeCallback(typeof(Action))]
        private static void OnApplicationEnterBackgroundHandler()
        {
            OnApplicationEnterBackgroundFromNative?.Invoke();
        }

        [AOT.MonoPInvokeCallback(typeof(Action))]
        private static void OnUserDidTakeScreenshotHandler()
        {
            OnUserDidTakeScreenshotFromNative?.Invoke();
        }

        internal override void Setup(TaskScheduler taskScheduler)
        {
            _magifySetup();
            _featuresProvider.Setup(taskScheduler, _magifyGetDefaultFeatures(), _magifyGetCurrentFeatures());
        }

        internal override void Reset()
        {
            _magifyReset();
        }

        internal override void ResetAndForceUpdate()
        {
            _magifyResetAndForceUpdate();
        }

        internal override void TweakAnalyticsConfig(int eventsGroupSize, int syncTimeInterval)
        {
            _magifyTweakAnalyticsConfig(eventsGroupSize, syncTimeInterval);
        }

        internal override void ResetAnalyticsConfig()
        {
            _magifyResetAnalyticsConfig();
        }

        internal override void TweakUserLocale(string languageTag)
        {
            _magifyTweakUserLocale(languageTag);
        }

        [AOT.MonoPInvokeCallback(typeof(Action))]
        private static void FeaturesUpdatedFromIOSPlatform()
        {
            OnFeaturesUpdatedFromNative?.Invoke();
        }

        [AOT.MonoPInvokeCallback(typeof(Action))]
        private static void CampaignsLoadedFromIOSPlatform()
        {
            _logger.Log($"Config loaded callback has been called");
            OnConfigLoadedFromNative?.Invoke();
        }

        internal override void Sync()
        {
            _magifyUpdate();
        }

        internal override void SetMediaSource(string networkName = null, string campaignName = null, string adGroup = null)
        {
            _magifySetMediaSource(networkName, campaignName, adGroup);
        }

        internal override void SetAttStatus(bool authorized)
        {
            _logger.Log($"Set att status to '{authorized}'");
            _magifySetATTStatus(authorized);
        }

        internal override void CompleteOffer(string offerName)
        {
            _magifyCompleteLtoWithCampaignName(offerName);
        }

        internal override void SetupConversionTracker(string revenuePerCountryPath, string revenueLevelsPath, string defaultCurrencyRatesPath, string subscriptionMultipliersPath)
        {
            _logger.Log($"Setup conversion tracker.\n{nameof(revenuePerCountryPath)}: {revenuePerCountryPath}\n{nameof(revenueLevelsPath)}: {revenueLevelsPath}\n{nameof(defaultCurrencyRatesPath)}: {defaultCurrencyRatesPath}\n{nameof(subscriptionMultipliersPath)}: {subscriptionMultipliersPath}");
            _magifySetupConversionTracker(revenuePerCountryPath, revenueLevelsPath, defaultCurrencyRatesPath, subscriptionMultipliersPath);
        }

        internal override ICampaign CampaignFor(string eventName, Dictionary<string, object> customParams = null, bool silent = false)
        {
            var paramsString = JsonFacade.SerializeObject(customParams, _jsonSerializerSettings);
            var pointer = _magifyGetCampaignFor(eventName, paramsString, silent);
            var model = (CampaignModel) Marshal.PtrToStructure(pointer, typeof(CampaignModel));
            var campaign = GetCampaign(model);
            _magifyFreeCampaign(pointer);

            return campaign;
        }

        internal override bool IsCampaignAvailable(string campaignName, string eventName, Dictionary<string, object> customParams = null)
        {
            var paramsString = JsonFacade.SerializeObject(customParams, _jsonSerializerSettings);
            return _magifyIsCampaignAvailable(campaignName, eventName, paramsString);
        }

        private static ICampaign GetCampaign(CampaignModel model)
        {
            switch (model.Type)
            {
                case CampaignType.Banner:
                    var bannerPointer = _magifyGetCurrentBannerCampaign();
                    var bannerModel = (BannerCampaignModel)Marshal.PtrToStructure(bannerPointer, typeof(BannerCampaignModel));

                    var bannerCampaign = new BannerCampaign
                    {
                        Name = bannerModel.Name,
                        Position = bannerModel.Position
                    };

                    _magifyFreeBannerCampaign(bannerPointer);
                    return bannerCampaign;

                case CampaignType.RewardedVideo:
                    var rewardedPointer = _magifyGetCurrentRewardedCampaign();
                    var rewardedCampaign = GetRewardedVideoCampaignFrom(rewardedPointer, model);
                    _magifyFreeRewardedCampaign(rewardedPointer);

                    return rewardedCampaign;
                case CampaignType.LtoRewardedVideo:
                    var rewardedLtoPointer = _magifyGetCurrentRewardedLtoCampaign();
                    var rewardedLtoCampaign = GetRewardedVideoCampaignFrom(rewardedLtoPointer, model);
                    _magifyFreeRewardedCampaign(rewardedLtoPointer);

                    return rewardedLtoCampaign;
                case CampaignType.Interstitial:
                    var interPointer = _magifyGetCurrentInterstitialCampaign();
                    var interModel = (InterstitialCampaignModel)Marshal.PtrToStructure(interPointer, typeof(InterstitialCampaignModel));
                    if(!Enum.IsDefined(typeof(CampaignType), interModel.Type))
                    {
                        return null;
                    }

                    var screenAttributes = Json.Deserialize(interModel.SplashScreen.Attributes) as Dictionary<string, object>;
                    var interstitialCampaign = new InterstitialCampaign
                    {
                        Name = interModel.Name,
                        Screen = new InterScreen
                        {
                            Type = interModel.SplashScreen.Type,
                            MinTime = interModel.SplashScreen.MinTime,
                            MaxTime = interModel.SplashScreen.MaxTime,
                            ExactTime = interModel.SplashScreen.ExactTime,
                            Timeout = interModel.SplashScreen.Timeout,
                            IsForced = interModel.SplashScreen.IsForced,
                            Attributes = screenAttributes
                        }.ToSplashScreen()
                    };
                    _magifyFreeInterstitialCampaign(interPointer);

                    return interstitialCampaign;

                case CampaignType.Subscription:
                    var subsPointer = _magifyGetCurrentSubscriptionCampaign();
                    var subsCampaign = GetSubscriptionCampaignFrom(subsPointer);
                    _magifyFreeSubscriptionCampaign(subsPointer);

                    return subsCampaign;
                case CampaignType.LtoSubscription:
                    var subsLtoPointer = _magifyGetCurrentSubscriptionLtoCampaign();
                    var subsLtoCampaign = GetSubscriptionCampaignFrom(subsLtoPointer);
                    _magifyFreeSubscriptionCampaign(subsLtoPointer);

                    return subsLtoCampaign;
                case CampaignType.InApp:
                    var inAppPointer = _magifyGetCurrentInAppCampaign();
                    var inAppCampaign = GetInappCampaignFrom(inAppPointer);
                    _magifyFreeInAppCampaign(inAppPointer);

                    return inAppCampaign;
                case CampaignType.LtoInApp:
                    var inAppLtoPointer = _magifyGetCurrentInAppLtoCampaign();
                    var inAppLtoCampaign = GetInappCampaignFrom(inAppLtoPointer);
                    _magifyFreeInAppCampaign(inAppLtoPointer);

                    return inAppLtoCampaign;
                case CampaignType.CrossPromo:
                    var promoPointer = _magifyGetCurrentPromoCampaign();
                    var promoCampaign = GetPromoOrExternalCampaignFrom(promoPointer);
                    _magifyFreePromoOrExternalCampaign(promoPointer);

                    return promoCampaign;
                case CampaignType.External:
                    var externalPointer = _magifyGetCurrentExternalCampaign();
                    var externalCampaign = GetPromoOrExternalCampaignFrom(externalPointer);
                    _magifyFreePromoOrExternalCampaign(externalPointer);

                    return externalCampaign;
                case CampaignType.LtoCrossPromo:
                    var ltoPromoPointer = _magifyGetCurrentPromoLtoCampaign();
                    var ltoPromoCampaign = GetPromoOrExternalCampaignFrom(ltoPromoPointer);
                    _magifyFreePromoOrExternalCampaign(ltoPromoPointer);

                    return ltoPromoCampaign;
                case CampaignType.LtoExternal:
                    var ltoExternalPointer = _magifyGetCurrentExternalLtoCampaign();
                    var ltoExternalCampaign = GetPromoOrExternalCampaignFrom(ltoExternalPointer);
                    _magifyFreePromoOrExternalCampaign(ltoExternalPointer);

                    return ltoExternalCampaign;
                case CampaignType.RateReview:
                    var reviewCampaign = new RateReviewCampaign
                    {
                        Name = model.Name
                    };
                    return reviewCampaign;
                case CampaignType.Notification:
                    var notificationPointer = _magifyGetCurrentNotificationCampaign();
                    var notificationModel = (NotificationCampaignModel)Marshal.PtrToStructure(notificationPointer, typeof(NotificationCampaignModel));
                    var notificationScreenAttributes = Json.Deserialize(notificationModel.screen.Attributes) as Dictionary<string, object>;
                    var notificationCampaign = new NotificationCampaign
                    {
                        Name = notificationModel.name,
                        Creative = new CampaignScreen
                        {
                            Type = notificationModel.screen.Type,
                            ScreenId = notificationModel.screen.ScreenID,
                            Attributes = notificationScreenAttributes,
                            PortraitUrl = notificationModel.screen.PortraitUrl,
                            LandscapeUrl = notificationModel.screen.LandscapeUrl,
                            ButtonCloseTimeout = (float)notificationModel.screen.ButtonCloseTimeout,
                            Color = notificationModel.screen.Color.ToColor(),
                            Opacity = (float)notificationModel.screen.Opacity,
                            BlurBackground = notificationModel.screen.BlurBackground
                        }.ToCreative(),
                    };
                    _magifyFreeNotificationCampaign(notificationPointer);

                    return notificationCampaign;
                case CampaignType.Bonus:
                    var bonusPointer = _magifyGetCurrentBonusCampaign();
                    var bonusCampaign = GetBonusCampaignFrom(bonusPointer);
                    _magifyFreeBonusCampaign(bonusPointer);

                    return bonusCampaign;
                case CampaignType.LtoBonus:
                    var bonusLtoPointer = _magifyGetCurrentBonusLtoCampaign();
                    var bonusLtoCampaign = GetBonusCampaignFrom(bonusLtoPointer);
                    _magifyFreeBonusCampaign(bonusLtoPointer);

                    return bonusLtoCampaign;
                case CampaignType.Mixed:
                    var mixedPointer = _magifyGetCurrentMixedCampaign();
                    var mixedCampaign = GetMixedCampaignFrom(mixedPointer);
                    _magifyFreeMixedCampaign(mixedPointer);

                    return mixedCampaign;
                case CampaignType.LtoMixed:
                    var mixedLtoPointer = _magifyGetCurrentMixedLtoCampaign();
                    var mixedLtoCampaign = GetMixedCampaignFrom(mixedLtoPointer);
                     _magifyFreeMixedCampaign(mixedLtoPointer);

                    return mixedLtoCampaign;
                default:
                    return null;
            }
        }

        internal override CampaignImpression LastImpressionFor(CampaignType campaignType, string campaignName, string eventName)
        {
            var impressionPointer = _magifyLastImpressionFor((int) campaignType, campaignName, eventName);
            var impressionModel = (ImpressionModel) Marshal.PtrToStructure(impressionPointer, typeof(ImpressionModel));
            var attributesDict = Json.Deserialize(impressionModel.Attributes) as Dictionary<string, object>;
            _magifyFreeImpression(impressionPointer);

            var impression = new CampaignImpression
            {
                CampaignName = impressionModel.CampaignName,
                ImpressionNumber = impressionModel.ImpressionNumber,
                SessionImpressionNumber = impressionModel.SessionImpressionNumber,
                CampaignTypeImpressionNumber = impressionModel.CampaignTypeImpressionNumber,
                CampaignTypeSessionImpressionNumber = impressionModel.CampaignTypeSessionImpressionNumber,

                EventName = impressionModel.EventName,
                EventNumber = impressionModel.EventNumber,
                SessionEventNumber = impressionModel.SessionEventNumber,
                Parameters = attributesDict
            };

            return impression;
        }

        internal override void SubscribeCampaignUpdates(string campaignName, Action<ICampaign> onUpdate)
        {
            CampaginUpdateCallbacks[campaignName] = onUpdate;
            _magifySubscribeCampaignUpdates(campaignName, CampaignDidUpdateHandler);
        }

        [AOT.MonoPInvokeCallback(typeof(Action<IntPtr>))]
        private static void CampaignDidUpdateHandler(IntPtr pointer)
        {
            var campaignModel = (CampaignModel)Marshal.PtrToStructure(pointer, typeof(CampaignModel));
            if (CampaginUpdateCallbacks.ContainsKey(campaignModel.Name)) {
                CampaginUpdateCallbacks[campaignModel.Name]?.Invoke(GetCampaign(campaignModel));
            }

            _magifyFreeCampaign(pointer);
        }

        internal override void UnsubscribeCampaignUpdates(string campaignName)
        {
            CampaginUpdateCallbacks.Remove(campaignName);
            _magifyUnsubscribeCampaignUpdates(campaignName);
        }

        internal override bool HasProcessedPurchase(string productId)
        {
            return _magifyHasProcessedPurchaseFor(productId);
        }

        internal override void SetGameMode(string mode)
        {
            _magifySetGameMode(mode);
        }

        internal override void SetGameLevel(int level)
        {
            _magifySetGameLevel(level);
        }

        internal override void SetGameMaxLevel(int maxLevel)
        {
            _magifySetGameMaxLevel(maxLevel);
        }

        internal override void SetUserEmail(string email)
        {
            _magifySetUserEmail(email);
        }

        internal override void TrackGdprAccessState(bool accessState)
        {
        }

        internal override void TrackAppLaunch()
        {
            _magifyTrackAppLaunch();
        }

        internal override void TrackCustomEvent(string eventName, IReadOnlyDictionary<string, object> customParams = null)
        {
            var paramsString = JsonFacade.SerializeObject(customParams, _jsonSerializerSettings);
            _magifyTrackCustomEventFor(eventName, paramsString);
        }

        internal override void TrackImpression(CampaignType campaignType)
        {
            _magifyTrackImpressionFor((int) campaignType);
        }

        internal override void TrackParentCampaignImpression(CampaignType campaignType)
        {
            _magifyTrackParentImpressionFor((int)campaignType);
        }

        internal override void TrackProductsImpression(CampaignType campaignType, List<string> productIds)
        {
            _magifyTrackNestedImpressionFor((int)campaignType, string.Join(",", productIds));
        }

        internal override void TrackAdsImpression(CampaignType campaignType, IAdsImpression impression)
        {
            var impressionString = JsonFacade.SerializeObject(impression, _jsonSerializerSettings);
            _magifyTrackAdsImpressionWithInfoFor((int) campaignType, impressionString);
        }

        internal override void TrackImpressionFailFor(CampaignType campaignType, string reason)
        {
            _magifyTrackImpressionFailFor((int) campaignType, reason);
        }

        internal override void TrackClickFor(CampaignType campaignType)
        {
            _magifyTrackClickFor((int) campaignType);
        }

        internal override void TrackAdsClickFor(CampaignType campaignType)
        {
            _magifyTrackAdsClickFor((int)campaignType);
        }

        internal override void TrackProductClickFor(CampaignType campaignType, string productId)
        {
            _magifyTrackPurchaseClickFor((int) campaignType, productId);
        }

        internal override void TrackAdsProductClickFor(CampaignType campaignType, string productId)
        {
            _magifyTrackAdsPurchaseClickFor((int)campaignType, productId);
        }

        internal override void TrackSubscriptionActivation(bool isTrial, string productId, string price, string currency, string period, string transactionId, string purchaseToken, string originalTransactionId, string receipt)
        {
            _magifyTrackSubscriptionActivation(isTrial, productId, price, currency, period, transactionId, originalTransactionId);
        }

        internal override void TrackExternalSubscriptionActivation(bool isTrial, string productId, string price, string currency, string transactionId, string purchaseToken, string originalTransactionId, string receipt)
        {
            _magifyTrackExternalSubscriptionActivation(isTrial, productId, price, currency, transactionId, originalTransactionId);
        }

        internal override void TrackInAppFor(string productId, string price, string currency, string transactionId, string purchaseToken, string originalTransactionId, string receipt)
        {
            _magifyTrackInAppFor(productId, price, currency, transactionId, originalTransactionId);
        }

        internal override void TrackExternalInAppFor(string productId, string price, string currency, string transactionId, string purchaseToken, string originalTransactionId, string receipt)
        {
            _magifyTrackExternalInAppFor(productId, price, currency, transactionId, originalTransactionId);
        }

        internal override void TrackRestoredInAppFor(string productId)
        {
            _magifyTrackRestoredInAppFor(productId);
        }

        internal override void TrackExpenseTransaction(List<BonusInfo> bonuses)
        {
            _magifyTrackExpenseTransaction(JsonFacade.SerializeObject(bonuses, _jsonSerializerSettings));
        }

        internal override void TrackIncomeTransaction(string source, List<BonusInfo> bonuses, ProductInfo product = null)
        {
            _magifyTrackIncomeTransaction(source, JsonFacade.SerializeObject(bonuses, _jsonSerializerSettings), product?.ProductId, product?.Price, product?.Currency);
        }

        internal override void TrackCorrectionTransaction(List<BonusInfo> bonuses)
        {
            _logger.LogError($"Correction transactions are not supported in current version");
        }

        internal override void TrackRewardGranted(string productId)
        {
            _magifyTrackRewardGranted(productId);
        }

        internal override void TrackFreeBonusGranted(string productId)
        {
            _magifyTrackRewardGranted(productId);
        }

        internal override void TrackOrdinaryProductUsed(string productId)
        {
            _magifyTrackOrdinaryProduct(productId);
        }

        internal override List<ContentItem> GetContentList(string group, string key, List<string> tags = null)
        {
            var itemsJson = _magifyGetContentListForGroupKey(group, key, JsonFacade.SerializeObject(tags, _jsonSerializerSettings));
            var rawContentList = (List<object>) Json.Deserialize(itemsJson);
            var items = new List<ContentItem>();

            foreach (var itemJson in rawContentList)
            {
                var itemStr = itemJson as string;
                var item = ContentItemFromJSON(itemStr);
                items.Add(item);
            }

            return items;
        }

        internal override ContentItem GetEarliestContent(string group, string key)
        {
            var itemJson = _magifygetEarliestContentForGroupKey(group, key);
            return itemJson != null ? ContentItemFromJSON(itemJson) : null;
        }

        internal override ContentItem GetLatestContent(string group, string key)
        {
            var itemJson = _magifygetLatestContentForGroupKey(group, key);
            return itemJson != null ? ContentItemFromJSON(itemJson) : null;
        }

        public override void Dispose()
        {
            // ToDo: try to free all resources and subscriptions
        }

        private static RewardedVideoCampaign GetRewardedVideoCampaignFrom(IntPtr pointer, CampaignModel model)
        {
            var rewardedModel = (RewardedVideoCampaignModel)Marshal.PtrToStructure(pointer, typeof(RewardedVideoCampaignModel));
            var rewardProducts = ParseRewardProducts(rewardedModel.products);
            var rewardedScreenAttributes = Json.Deserialize(rewardedModel.screen.Attributes) as Dictionary<string, object>;

            var rwCampaign = new RewardedVideoCampaign
            {
                Name = model.Name,
                Type = rewardedModel.type,
                Products = rewardProducts,
                Creative = new CampaignScreen
                {
                    Type = rewardedModel.screen.Type,
                    ScreenId = rewardedModel.screen.ScreenID,
                    Attributes = rewardedScreenAttributes,
                    PortraitUrl = rewardedModel.screen.PortraitUrl,
                    LandscapeUrl = rewardedModel.screen.LandscapeUrl,
                    ButtonCloseTimeout = (float)rewardedModel.screen.ButtonCloseTimeout,
                    Color = rewardedModel.screen.Color.ToColor(),
                    Opacity = (float)rewardedModel.screen.Opacity,
                    BlurBackground = rewardedModel.screen.BlurBackground
                }.ToCreative()
            };
            return rwCampaign;
        }

        private static SubscriptionCampaign GetSubscriptionCampaignFrom(IntPtr pointer)
        {
            var subsModel = (SubscriptionCampaignModel)Marshal.PtrToStructure(pointer, typeof(SubscriptionCampaignModel));
            if(!Enum.IsDefined(typeof(CampaignType), subsModel.Type))
            {
                return null;
            }

            var products = ParseSubscriptionProductsFromDict(subsModel.ProductIDs);
            var screenAttributes = Json.Deserialize(subsModel.screen.Attributes) as Dictionary<string, object>;
            var subsCampaign = new SubscriptionCampaign
            {
                Type = subsModel.Type,
                Name = subsModel.Name,
                Products = products,
                Creative = new CampaignScreen
                {
                    Type = subsModel.screen.Type,
                    ScreenId = subsModel.screen.ScreenID,
                    Attributes = screenAttributes,
                    PortraitUrl = subsModel.screen.PortraitUrl,
                    LandscapeUrl = subsModel.screen.LandscapeUrl,
                    ButtonCloseTimeout = (float)subsModel.screen.ButtonCloseTimeout,
                    Color = subsModel.screen.Color.ToColor(),
                    Opacity = (float)subsModel.screen.Opacity,
                    BlurBackground = subsModel.screen.BlurBackground
                }.ToCreative(),
            };

            return subsCampaign;
        }

        private static InAppCampaign GetInappCampaignFrom(IntPtr pointer)
        {
            var inAppModel = (InAppCampaignModel)Marshal.PtrToStructure(pointer, typeof(InAppCampaignModel));
            if(!Enum.IsDefined(typeof(CampaignType), inAppModel.type))
            {
                return null;
            }

            var inAppProducts = ParseInAppProducts(inAppModel.products);
            var screenAttributes = Json.Deserialize(inAppModel.screen.Attributes) as Dictionary<string, object>;
            var inAppCampaign = new InAppCampaign
            {
                Type = inAppModel.type,
                Name = inAppModel.name,
                Creative = new CampaignScreen
                {
                    Type = inAppModel.screen.Type,
                    ScreenId = inAppModel.screen.ScreenID,
                    Attributes = screenAttributes,
                    PortraitUrl = inAppModel.screen.PortraitUrl,
                    LandscapeUrl = inAppModel.screen.LandscapeUrl,
                    ButtonCloseTimeout = (float)inAppModel.screen.ButtonCloseTimeout,
                    Color = inAppModel.screen.Color.ToColor(),
                    Opacity = (float)inAppModel.screen.Opacity,
                    BlurBackground = inAppModel.screen.BlurBackground
                }.ToCreative(),
                Products = inAppProducts
            };

            return inAppCampaign;
        }

        private static BonusCampaign GetBonusCampaignFrom(IntPtr pointer)
        {
            var bonusModel = (BonusCampaignModel)Marshal.PtrToStructure(pointer, typeof(BonusCampaignModel));
            if(!Enum.IsDefined(typeof(CampaignType), bonusModel.type))
            {
                return null;
            }

            var bonusProducts = ParseBonusProducts(bonusModel.products);
            var screenAttributes = Json.Deserialize(bonusModel.screen.Attributes) as Dictionary<string, object>;

            var bonusCampaign = new BonusCampaign
            {
                Type = bonusModel.type,
                Name = bonusModel.name,
                Creative = new CampaignScreen
                {
                    Type = bonusModel.screen.Type,
                    ScreenId = bonusModel.screen.ScreenID,
                    Attributes = screenAttributes,
                    PortraitUrl = bonusModel.screen.PortraitUrl,
                    LandscapeUrl = bonusModel.screen.LandscapeUrl,
                    ButtonCloseTimeout = (float)bonusModel.screen.ButtonCloseTimeout,
                    Color = bonusModel.screen.Color.ToColor(),
                    Opacity = (float)bonusModel.screen.Opacity,
                    BlurBackground = bonusModel.screen.BlurBackground
                }.ToCreative(),
                Products = bonusProducts
            };

            return bonusCampaign;
        }

        private static MixedCampaign GetMixedCampaignFrom(IntPtr pointer)
        {
            var mixedModel = (MixedCampaignModel)Marshal.PtrToStructure(pointer, typeof(MixedCampaignModel));
            if(!Enum.IsDefined(typeof(CampaignType), mixedModel.type))
            {
                return null;
            }

            var mixedProducts = ParseMixedProducts(mixedModel.products);
            var screenAttributes = Json.Deserialize(mixedModel.screen.Attributes) as Dictionary<string, object>;

            var mixedCampaign = new MixedCampaign
            {
                Type = mixedModel.type,
                Name = mixedModel.name,
                Creative = new CampaignScreen
                {
                    Type = mixedModel.screen.Type,
                    ScreenId = mixedModel.screen.ScreenID,
                    Attributes = screenAttributes,
                    PortraitUrl = mixedModel.screen.PortraitUrl,
                    LandscapeUrl = mixedModel.screen.LandscapeUrl,
                    ButtonCloseTimeout = (float)mixedModel.screen.ButtonCloseTimeout,
                    Color = mixedModel.screen.Color.ToColor(),
                    Opacity = (float)mixedModel.screen.Opacity,
                    BlurBackground = mixedModel.screen.BlurBackground
                }.ToCreative(),
                Products = mixedProducts
            };

            return mixedCampaign;
        }

        private static ICampaign GetPromoOrExternalCampaignFrom(IntPtr pointer)
        {
            var promoModel = (PromoCampaignModel)Marshal.PtrToStructure(pointer, typeof(PromoCampaignModel));
            var promoScreenAttributes = Json.Deserialize(promoModel.screen.Attributes) as Dictionary<string, object>;

            var creative = new CampaignScreen
            {
                Type = promoModel.screen.Type,
                ScreenId = promoModel.screen.ScreenID,
                Attributes = promoScreenAttributes,
                PortraitUrl = promoModel.screen.PortraitUrl,
                LandscapeUrl = promoModel.screen.LandscapeUrl,
                ButtonCloseTimeout = (float)promoModel.screen.ButtonCloseTimeout,
                Color = promoModel.screen.Color.ToColor(),
                Opacity = promoModel.screen.Opacity,
                BlurBackground = promoModel.screen.BlurBackground
            }.ToCreative();

            return promoModel.DestinationType switch
            {
                DestinationType.Application => new CrossPromoCampaign
                {
                    Type = promoModel.Type,
                    Creative = creative,
                    Name = promoModel.Name,
                    Products = new List<CrossPromoProduct>
                    {
                        new(ProductDef.FakeProductId, promoModel.BundleId, promoModel.RawPromoLink, promoModel.Schemas.Split(',', StringSplitOptions.RemoveEmptyEntries), null, creative.Attributes)
                    }
                },
                DestinationType.Link => new ExternalPromoCampaign
                {
                    Type = promoModel.Type,
                    Creative = creative,
                    Name = promoModel.Name,
                    Products = new List<ExternalLinkProduct>
                    {
                        new(ProductDef.FakeProductId, promoModel.DestinationValue, null, creative.Attributes)
                    }
                },
                _ => null
            };
        }

        private static List<InAppProduct> ParseInAppProducts(string productsString)
        {
            var inAppProductsDict = (Dictionary<string, object>) Json.Deserialize(productsString);
            var productsModel = (string) inAppProductsDict["products"];
            return ParseInAppProductsList(productsModel);
        }

        private static List<InAppProduct> ParseInAppProductsList(string productsListString)
        {
            var productsList = (List<object>) Json.Deserialize(productsListString);
            var inAppProducts = new List<InAppProduct>();
            foreach (var product in productsList)
            {
                var productStr = (string) product;
                var productDict = (Dictionary<string, object>) Json.Deserialize(productStr);
                var productId = (string) productDict["productId"];
                var productType = (ProductType) int.Parse(productDict["productType"].ToString());
                var context = Json.Deserialize((string) productDict["context"]) as Dictionary<string, object>;
                var inAppProduct = new InAppProduct(productId, productType == ProductType.Consumable, null, context);
                inAppProducts.Add(inAppProduct);
            }
            return inAppProducts;
        }

        private static List<BonusProduct> ParseBonusProducts(string productsString)
        {
            var inAppProductsDict = (Dictionary<string, object>)Json.Deserialize(productsString);
            var productsModel = (string)inAppProductsDict["products"];
            return ParseBonusProductsList(productsModel);
        }

        private static List<BonusProduct> ParseBonusProductsList(string productsListString)
        {
            var productsList = (List<object>)Json.Deserialize(productsListString);
            var bonusProducts = new List<BonusProduct>();
            foreach (var product in productsList)
            {
                var productStr = (string)product;
                var productDict = (Dictionary<string, object>)Json.Deserialize(productStr);
                var productId = (string)productDict["productId"];
                var context = Json.Deserialize((string)productDict["context"]) as Dictionary<string, object>;
                var bonusProduct = new BonusProduct(productId, null, context);
                bonusProducts.Add(bonusProduct);
            }
            return bonusProducts;
        }

        private static List<ProductDef> ParseMixedProducts(string productsString)
        {
            var mixedProductsDict = (Dictionary<string, object>)Json.Deserialize(productsString);
            var productsModel = (string)mixedProductsDict["products"];
            return ParseMixedProductsList(productsModel);
        }

        private static List<ProductDef> ParseMixedProductsList(string productsListString)
        {
            var productsList = (List<object>)Json.Deserialize(productsListString);
            var mixedProducts = new List<ProductDef>();
            foreach (var product in productsList)
            {
                var productStr = (string)product;
                var productDict = (Dictionary<string, object>)Json.Deserialize(productStr);
                var productId = (string)productDict["productId"];
                var productType = (ProductType)int.Parse(productDict["productType"].ToString());
                var context = Json.Deserialize((string)productDict["context"]) as Dictionary<string, object>;
                var count = int.Parse(productDict["count"].ToString());

                ProductCreative creative = null;
                if (productDict.TryGetValue("creative", out var creativeVal))
                {
                    creative = ParseCardCreative((string)creativeVal);
                }

                var link = string.Empty;
                if (productDict.TryGetValue("link", out var linkVal))
                {
                    link = (string)linkVal;
                }

                string promoUrl = null;
                if (productDict.TryGetValue("promoUrl", out var promoUrlVal))
                {
                    promoUrl = (string)promoUrlVal;
                }

                string bundleId = null;
                if (productDict.TryGetValue("bundleId", out var bundleIdVal))
                {
                    bundleId = (string)bundleIdVal;
                }

                List<string> schemas = null;
                if (productDict.TryGetValue("schemas", out var schemasVal))
                {
                    var schemasArray = ((string)schemasVal).Split(',');
                    schemas = new List<string>(schemasArray);
                }

                switch (productType)
                {
                    case ProductType.Consumable:
                        mixedProducts.Add(new InAppProduct(productId, true, creative, context));
                        break;
                    case ProductType.NonConsumable:
                        mixedProducts.Add(new InAppProduct(productId, false, creative, context));
                        break;
                    case ProductType.Bonus:
                        mixedProducts.Add(new BonusProduct(productId, creative, context));
                        break;
                    case ProductType.Rewarded:
                        mixedProducts.Add(new RewardProduct(productId, count, creative, context));
                        break;
                    case ProductType.Subscription:
                        mixedProducts.Add(new SubscriptionProduct(productId, creative, context));
                        break;
                    case ProductType.Info:
                        mixedProducts.Add(new InfoProduct(productId, creative, context));
                        break;
                    case ProductType.ExternalLink:
                        mixedProducts.Add(new ExternalLinkProduct(productId, link, creative, context));
                        break;
                    case ProductType.InternalLink:
                        mixedProducts.Add(new InternalLinkProduct(productId, link, creative, context));
                        break;
                    case ProductType.CrossPromo:
                        mixedProducts.Add(new CrossPromoProduct(productId, bundleId, promoUrl, schemas, creative, context));
                        break;
                }
            }
            return mixedProducts;
        }

        private static ProductCreative ParseCardCreative(string creativeStr)
        {
            var dictionary = (Dictionary<string, object>)Json.Deserialize(creativeStr);
            var attributesDict = Json.Deserialize((string)dictionary["attributes"]) as Dictionary<string, object>;
            return new ProductCreative
            {
                Title = (string)dictionary["title"],
                Text = (string)dictionary["text"],
                Resource = ParseCreativeResource((string)dictionary["resource"]),
                ActionButton = (string)dictionary["actionButton"],
                CloseButton = (string)dictionary["closeButton"],
                Attributes = attributesDict
            };
        }

        private static List<RewardProduct> ParseRewardProducts(string productsString)
        {
            var inAppProductsDict = (Dictionary<string, object>)Json.Deserialize(productsString);
            var productsModel = (string)inAppProductsDict["products"];
            return ParseRewardProductsList(productsModel);
        }

        private static List<RewardProduct> ParseRewardProductsList(string productsListString)
        {
            var productsList = (List<object>)Json.Deserialize(productsListString);
            var rewardProducts = new List<RewardProduct>();
            foreach (var product in productsList)
            {
                var productStr = (string)product;
                var productDict = (Dictionary<string, object>)Json.Deserialize(productStr);
                var productId = (string)productDict["productId"];
                var context = Json.Deserialize((string)productDict["context"]) as Dictionary<string, object>;
                var count = int.Parse(productDict["count"].ToString());
                var rewardProduct = new RewardProduct(productId, count, null, context);
                rewardProducts.Add(rewardProduct);
            }
            return rewardProducts;
        }

        private static List<SubscriptionProduct> ParseSubscriptionProductsFromDict(string productsDictString)
        {
            var productDict = (Dictionary<string, object>)Json.Deserialize(productsDictString);
            var productList = (string)productDict["products"];
            return ParseSubscriptionProductsList(productList);
        }

        private static List<SubscriptionProduct> ParseSubscriptionProductsList(string productsListString)
        {
            var productsList = (List<object>) Json.Deserialize(productsListString);
            var subProducts = new List<SubscriptionProduct>();
            foreach (var product in productsList)
            {
                var productStr = product as string;
                var productDict = (Dictionary<string, object>) Json.Deserialize(productStr);
                var productId = (string) productDict["productId"];
                var context = Json.Deserialize(productDict["context"] as string) as Dictionary<string, object>;
                var subProduct = new SubscriptionProduct(productId, null, context);
                subProducts.Add(subProduct);
            }
            return subProducts;
        }

        private ContentItem ContentItemFromJSON(string json)
        {
            var contentItemDict = (Dictionary<string, object>) Json.Deserialize(json);
            var itemKey = (string) contentItemDict["key"];
            var itemLink = (string) contentItemDict["link"];
            var itemPreviews = ((List<object>) contentItemDict["previews"]).ConvertAll(x => x.ToString());
            var itemTags = ((List<object>) contentItemDict["tags"]).ConvertAll(x => x.ToString());

            var item = new ContentItem
            {
                Key = itemKey,
                Link = itemLink,
                Previews = itemPreviews,
                Tags = itemTags
            };
            return item;
        }

        private static IReadOnlyCollection<LtoInfo> LtoInfosFromJSON(string json)
        {
            var ltoInfoDict = (Dictionary<string, object>)Json.Deserialize(json);
            var ltoInfosSet = (string)ltoInfoDict["activeLtoOffers"];
            return ParseLtoInfosHashSet(ltoInfosSet);
        }

        private static IReadOnlyCollection<LtoInfo> ParseLtoInfosHashSet(string ltoInfos)
        {
            var ltoList = (List<object>)Json.Deserialize(ltoInfos);
            var ltoInfosHashSet = new List<LtoInfo>();

            foreach (var lto in ltoList)
            {
                var ltoStr = (string)lto;
                var ltoDict = (Dictionary<string, object>)Json.Deserialize(ltoStr);
                var spot = (string)ltoDict["spot"];
                var campaignName = (string)ltoDict["campaignName"];
                var startTime = UnixTimestamp.ToDateTime(int.Parse(ltoDict["startTime"].ToString()));
                var endTime = UnixTimestamp.ToDateTime(int.Parse(ltoDict["endTime"].ToString()));
                var badgeCreative = (string)ltoDict["badgeCreative"];

                var ltoInfo = new LtoInfo
                {
                    CampaignName = campaignName,
                    Spot = spot,
                    StartTime = startTime,
                    EndTime = endTime,
                    BadgeCreative = ParseBadgeCreative(badgeCreative)
                };

                ltoInfosHashSet.Add(ltoInfo);
            }
            return ltoInfosHashSet;
        }

        private static LtoBadgeCreative ParseBadgeCreative(string badgeCreativeStr)
        {
            var badgeDict = (Dictionary<string, object>)Json.Deserialize(badgeCreativeStr);
            var tag = (string)badgeDict["tag"];
            var title = (string)badgeDict["title"];
            var labelText = (string)badgeDict["labelText"];
            var resource = ParseCreativeResource((string)badgeDict["resource"]);
            var placeholder = (string)badgeDict["placeholder"];
            var hasTimer = bool.Parse(badgeDict["hasTimer"].ToString());
            var attributes = Json.Deserialize(badgeDict["attributes"] as string) as Dictionary<string, object>;

            var badgeCreative = new LtoBadgeCreative
            {
                Tag = tag,
                Title = title,
                LabelText = labelText,
                Resource = resource,
                Placeholder = placeholder,
                HasTimer = hasTimer,
                LastMinutesTimer = int.Parse(badgeDict["lastMinutesTimer"].ToString()),
                Attributes = attributes
            };

            return badgeCreative;
        }

        private static CreativeResource ParseCreativeResource(string resourceStr)
        {
            var resourceDict = (Dictionary<string, object>)Json.Deserialize(resourceStr);
            if (resourceDict.Count == 0)
            {
                return null;
            }

            var type = (CreativeResource.ArtifactType)int.Parse(resourceDict["type"].ToString());
            var url = (string)resourceDict["url"];

            return new CreativeResource
            {
                Url = url,
                Type = type
            };
        }
    }
}

#endif
