using System;
using System.Collections.Generic;
using System.Linq;
using JetBrains.Annotations;
using Magify.Rx;
using UnityEngine;

namespace Magify
{
    internal class MagifyEditorProxy : IDisposable
    {
        public event Action<LtoInfo> OnOfferAdded;
        public event Action<LtoInfo> OnOfferRemoved;
        public event Action<LtoInfo> OnOfferFinished;

        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Editor);

        [NotNull]
        private readonly IMagifyClient _client;
        private readonly bool _isDeveloperMode;
        [CanBeNull]
        private readonly MagifyEditorMoq _moq;
        [CanBeNull]
        private readonly EditorDevice _editorDevice;
        [CanBeNull]
        private readonly HashSet<CampaignType> _campaignsFromMoq;
        [NotNull]
        private readonly PooledCompositeDisposable _disposables = new();

        [NotNull]
        public IReadOnlyCollection<LtoInfo> ActiveLtoOffers => _moq?.ActiveLtoOffers() ?? (_isDeveloperMode ? ArraySegment<LtoInfo>.Empty : _client.GetActiveLtoOffers());

        public MagifyEditorProxy(
            [NotNull] IMagifyClient client,
            bool isDeveloperMode,
            [CanBeNull] MagifyEditorMoq moq,
            [CanBeNull] EditorDevice editorDevice)
        {
            _client = client;
            _isDeveloperMode = isDeveloperMode;
            _moq = moq;
            _editorDevice = editorDevice;
        }

        public void Initialize()
        {
            ScheduleRemoveActiveLtoAfterFinish();
        }

        public void Dispose()
        {
            _disposables.Release();
        }

        public void Reset()
        {
            _disposables.Clear();
        }

        public ICampaign CampaignFor(string eventName, IReadOnlyDictionary<string,object> customParams, bool silent)
        {
            if (eventName == "offer_finished")
            {
                var registeredLtoOffers = _moq?.RegisteredLtoOffers();
                if (customParams != null && registeredLtoOffers != null)
                {
                    if (customParams["campaign"] is string parentCampaign)
                    {
                        if (registeredLtoOffers.TryGetValue(parentCampaign, out var ltoInfo) && ltoInfo != null)
                        {
                            AddLto(ltoInfo);
                            return null;
                        }
                    }
                }
            }

            var campaign = _moq?.CampaignFor(eventName, customParams, silent);
            if (campaign != null)
            {
                _campaignsFromMoq?.Add(campaign.Type);
                return campaign;
            }

            if (!_isDeveloperMode)
            {
                campaign = _client.CampaignFor(eventName, customParams, silent);
                if (campaign != null)
                {
                    _campaignsFromMoq?.Remove(campaign.Type);
                }
            }
            return campaign;
        }


        private void ScheduleRemoveActiveLtoAfterFinish()
        {
            foreach (var ltoInfo in ActiveLtoOffers ?? ArraySegment<LtoInfo>.Empty)
            {
                ScheduleRemoveLtoAfterFinish(ltoInfo);
            }
        }

        private void ScheduleRemoveLtoAfterFinish(LtoInfo ltoInfo)
        {
            if (ltoInfo != null)
            {
                Observable.Timer(ltoInfo.EndTime)
                .Subscribe(_ => CompleteOffer(ltoInfo.CampaignName))
                .AddTo(_disposables);
            }
        }

        private void AddLto([NotNull] LtoInfo ltoInfo)
        {
            var activeOffers = _moq?.ActiveLtoOffers();
            if (activeOffers == null)
            {
                _logger.LogError($"Cannot add LTO {ltoInfo.CampaignName} because Moq is not set");
                return;
            }

            var ltoDuration = ltoInfo.EndTime - ltoInfo.StartTime;
            ltoInfo.StartTime = DateTime.UtcNow;
            ltoInfo.EndTime = DateTime.UtcNow + ltoDuration;

            ScheduleRemoveLtoAfterFinish(ltoInfo);

            activeOffers.Add(ltoInfo);
            OnOfferAdded?.Invoke(ltoInfo);
        }

        public void CompleteOffer(string offerName)
        {
            var activeOffers = _moq?.ActiveLtoOffers();
            if (activeOffers == null)
            {
                _logger.LogError($"Cannot complete LTO {offerName} because Moq is not set");
                return;
            }

            var lto = activeOffers.FirstOrDefault(c => c?.CampaignName == offerName);
            if (lto == null) return;

            activeOffers.Remove(lto);
            OnOfferRemoved?.Invoke(lto);
            OnOfferFinished?.Invoke(lto);
        }

        public bool IsCampaignFromMoq(CampaignType campaignType)
        {
            return _campaignsFromMoq?.Contains(campaignType) is true;
        }
    }
}