using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Rx;
using UnityEngine;

namespace Magify
{
    internal class MagifyCSharp : MagifyPlatformAPI, IModernBridgePlatformAPI
    {
        private static readonly MagifyLogger _logger = MagifyLogger.Get();
        private readonly string _appName;
        private readonly string _configPath;
        [CanBeNull]
        private readonly CustomClientId _customClientId;
        private readonly string _rootPath;
        [CanBeNull]
        private readonly MagifyEditorProxy _editorProxy;
        private readonly EditorDevice _editorDevice;
        [NotNull]
        private readonly ClientStateConfig _clientStateConfig;
        private readonly bool _isDeveloperMode;
        private readonly bool _useMinimal;

        [NotNull]
        private IMagifyClient _client;

        [NotNull]
        internal IMagifyClient Client => _client;

        /// <inheritdoc cref="MagifyDocs.CtorWithMagifyCustomClientIdWasJustSetException"/>
        public MagifyCSharp(
            MagifyEditorMoq moq,
            bool isDeveloperMode,
            bool useMinimal,
            EditorDevice editorDevice,
            [NotNull] string appName,
            [NotNull] string configPath,
            [CanBeNull] CustomClientId customClientId,
            [NotNull] string rootPath,
            [NotNull] ClientStateConfig clientStateConfig,
            Environment environment)
        {
            _isDeveloperMode = isDeveloperMode;
            _useMinimal = useMinimal;
            _editorDevice = editorDevice;
            _appName = appName;
            _configPath = configPath;
            _customClientId = customClientId;
            _rootPath = rootPath;
            _clientStateConfig = clientStateConfig;
            _client = _useMinimal
                ? new MinimalMagifyClient(_appName, _configPath, _rootPath, _customClientId, environment, _editorDevice)
                : new MagifyClient(_appName, _configPath, _rootPath, _customClientId, clientStateConfig, environment, editorDevice);

#if UNITY_EDITOR
            _editorProxy = new MagifyEditorProxy(_client, isDeveloperMode, moq, editorDevice);
#endif

            // ToDo: think about unsubscription
            _client.SessionNumber.Subscribe(number => OnSessionChanged?.Invoke(number));
            _client.OnUserDidTakeScreenshot += () => OnUserDidTakeScreenshot?.Invoke();

            _client.SubscriptionStatus.Subscribe(_ => OnSubscriptionStatusChanged?.Invoke());
            _client.InAppStatus.Subscribe(_ => OnInAppStatusChanged?.Invoke());
            _client.AuthorizationStatus.Subscribe(_ => OnAuthorizationStatusChanged?.Invoke());
            _client.ReferrerId.Subscribe(_ => OnReferrerChanged?.Invoke());
            _client.OnConfigParsed += configKind => OnConfigParsed?.Invoke(configKind);
            _client.OnConfigParsedOnMainThread += configKind => OnConfigParsedOnMainThread?.Invoke(configKind);
            _client.OnConfigLoaded += () => OnConfigLoaded?.Invoke();
            _client.OnPurchasedProductsChanged += () => OnPurchasedProductsChanged?.Invoke();
            _client.OnApplicationEnterForeground += () => OnApplicationEnterForeground?.Invoke();
            _client.OnApplicationEnterBackground += () => OnApplicationEnterBackground?.Invoke();
        }

        #region MagifyPlatformAPI events

        internal override event Action<ConfigKind> OnConfigParsed;
        internal override event Action<ConfigKind> OnConfigParsedOnMainThread;

        internal override event Action OnConfigLoaded;

        internal override event Action<int> OnSessionChanged;

        internal override event Action OnUserDidTakeScreenshot;

        public event Action<LtoInfo> OnOfferAdded
        {
            add
            {
                if (!_isDeveloperMode) _client.OnOfferAdded += value;
                if (_editorProxy != null) _editorProxy.OnOfferAdded += value;
            }
            remove
            {
                if (!_isDeveloperMode) _client.OnOfferAdded -= value;
                if (_editorProxy != null) _editorProxy.OnOfferAdded -= value;
            }
        }

        public event Action<LtoInfo> OnOfferUpdated
        {
            add
            {
                if (!_isDeveloperMode) _client.OnOfferUpdated += value;
            }
            remove
            {
                if (!_isDeveloperMode) _client.OnOfferUpdated -= value;
            }
        }

        public event Action<LtoInfo> OnOfferRemoved
        {
            add
            {
                if (!_isDeveloperMode) _client.OnOfferRemoved += value;
                if (_editorProxy != null) _editorProxy.OnOfferRemoved += value;
            }
            remove
            {
                if (!_isDeveloperMode) _client.OnOfferRemoved -= value;
                if (_editorProxy != null) _editorProxy.OnOfferRemoved -= value;
            }
        }

        public event Action<LtoInfo> OnOfferFinished
        {
            add
            {
                if (!_isDeveloperMode) _client.OnOfferFinished += value;
                if (_editorProxy != null) _editorProxy.OnOfferFinished += value;
            }
            remove
            {
                if (!_isDeveloperMode) _client.OnOfferFinished -= value;
                if (_editorProxy != null) _editorProxy.OnOfferFinished -= value;
            }
        }

        internal override event Action OnSubscriptionStatusChanged;
        internal override event Action OnInAppStatusChanged;
        internal override event Action OnAuthorizationStatusChanged;
        internal override event Action OnReferrerChanged;
        internal override event Action OnPurchasedProductsChanged;

        internal override event Action OnApplicationEnterForeground;
        internal override event Action OnApplicationEnterBackground;

        #endregion

        #region MagifyPlatformAPI properties

        internal override Environment Environment
        {
            get => _client.Environment.Value;
            set => _client.Environment.Value = value;
        }

        [NotNull]
        internal override IReactiveProperty<IPurchaseVerificationHandler> ExternalPurchaseVerificationHandler => _client.ExternalPurchaseVerificationHandler;
        [NotNull]
        internal override IReactiveProperty<float> VerificationRetryInterval => _client.VerificationRetryInterval;

        internal override SubscriptionStatus SubscriptionStatus
        {
            get => _client.SubscriptionStatus.Value;
            set => _client.SubscriptionStatus.Value = value;
        }

        internal override InAppStatus InAppStatus
        {
            get => _client.InAppStatus.Value;
            set => _client.InAppStatus.Value = value;
        }

        // Do nothing special
        internal override bool IsLoggingEnabled { get; set; }

        internal override string ReferrerId
        {
            get => _client.ReferrerId.Value;
            set => _client.ReferrerId.Value = value;
        }

        internal override string AdjustId
        {
            get => _client.AdjustId.Value;
            set => _client.AdjustId.Value = value;
        }

        internal override string FirebaseInstanceId
        {
            get => _client.FirebaseInstanceId.Value;
            set => _client.FirebaseInstanceId.Value = value;
        }

        internal override AuthorizationStatus AuthorizationStatus
        {
            get => _client.AuthorizationStatus.Value;
            set => _client.AuthorizationStatus.Value = value;
        }

        internal override bool IsPortrait => _client.IsPortrait;
        internal override ContextSyncTime LastContextSyncTime => _client.LastContextSyncTime;
        [NotNull]
        internal override IAghanimApi Aghanim => _client.AghanimApi;
        internal override IFeatures Features => _client?.Features;

        internal override Limits Limits
        {
            get
            {
                var limits = _client.Limits;
                return new Limits
                {
                    GlobalInterval = limits.GlobalInterval?.TotalSeconds ?? 0,
                    SessionsInterval = limits.SessionsInterval?.TotalSeconds ?? 0,
                    InterstitialInterval = limits.InterstitialInterval?.TotalSeconds ?? 0,
                    RewardInterstitialInterval = limits.RewardInterstitialInterval?.TotalSeconds ?? 0,
                    InterstitialsPerSession = limits.InterstitialsPerSession ?? int.MaxValue,
                    ImpressionsPerSession = limits.ImpressionsPerSession ?? int.MaxValue
                };
            }
        }

        internal override string ClientId => _client.ClientId;
        internal override bool CustomClientIdWasSet => _client.CustomClientIdWasSet;
        internal override IReactiveProperty<TimeSpan> SessionsInterval => _client.SessionsInterval;
        internal override int SessionNumber => _client.SessionNumber.Value;
        internal override string FirstInstalledVersion => _client.FirstInstalledVersion;
        internal override DateTime FirstLaunchDate => _client.FirstLaunchDate;
        [NotNull]
        internal override IReactiveProperty<ConfigScope> RemoteContextScope => _client.RemoteContextScopes;
        internal override IReadOnlyList<InAppProduct> InAppProducts => _client.InAppProducts;
        internal override IReadOnlyList<SubscriptionProduct> SubscriptionProducts => _client.SubscriptionProducts;
        [NotNull]
        internal override IReadOnlyReactiveProperty<IReadOnlyList<string>> Segmentations => _client.Segmentations;
        [NotNull]
        internal override IReadOnlyReactiveProperty<IReadOnlyList<AssignedAbTest>> AssignedAbTests => _client.AssignedAbTests;
        [NotNull]
        internal override IStoredAppFeaturesCollection StoredAppFeatures => _client.StoredAppFeaturesProvider;

        internal override IReadOnlyCollection<LtoInfo> ActiveLtoOffers => _editorProxy == null ? _client.GetActiveLtoOffers() : _editorProxy.ActiveLtoOffers;
        [NotNull]
        internal override IReactiveProperty<bool> SyncStateEnabled => _client.SyncStateEnabled;
        [NotNull]
        internal override IUniTaskSource<bool> ClientIdFromCloudLoadingPromise => _client.ClientIdFromCloudLoadingPromise;
        [NotNull]
        internal override IObservable<SyncStateResult> OnRestoreStateCompleted => _client.OnRestoreStateCompleted;
        [NotNull]
        internal override IReadOnlyReactiveProperty<AutoRestoreStateInfo> AutoRestoreStateInfo => _client.AutoRestoreStateInfo;
        [NotNull]
        internal override IReactiveProperty<bool> IsAutoRestoreStateEnabled => _client.IsAutoRestoreStateEnabled;
        [NotNull]
        public override IReadOnlyReactiveProperty<long> LastSuccessfulSyncProgressUnixTimeSeconds => _client.LastSuccessfulSyncProgressUnixTimeSeconds;

        public override bool ShouldSendLastSyncProgressTime
        {
            get => _client.ShouldSendLastSyncProgressTime;
            set => _client.ShouldSendLastSyncProgressTime = value;
        }

        internal override bool HasSocialAuthorizationData => _client.HasSocialAuthorizationData;
        internal override bool SkipClientIdFromCloudLoading
        {
            get => _client.SkipClientIdFromCloudLoading;
            set => _client.SkipClientIdFromCloudLoading = value;
        }

        internal override uint ClientIdFromCloudLoadingSecondsTimeout
        {
            get => _client.ClientIdFromCloudLoadingSecondsTimeout;
            set => _client.ClientIdFromCloudLoadingSecondsTimeout = value;
        }

        internal override bool IsSdkInitialized => true;

        #endregion

        #region MagifyPlatformAPI

        internal override void InitializeSdk()
        {
            _editorProxy?.Initialize();
            _client.InitializeSdk();
        }

        internal override UniTask AwaitInitialConfigParsing(CancellationToken cancellationToken)
        {
            return _client.AwaitInitialConfigParsing(cancellationToken);
        }

        internal override void Reset(bool clearNativeStorage = true, bool clearCloudStorage = true)
        {
            _editorProxy?.Reset();
            var environment = _client.Environment.Value;
            _client.Reset(clearNativeStorage, clearCloudStorage);
            _client.Dispose();
            if (Directory.Exists(_rootPath))
            {
                Directory.Delete(_rootPath, true);
            }
            _client = _useMinimal
                ? new MinimalMagifyClient(_appName, _configPath, _rootPath, _customClientId, environment, _editorDevice)
                : new MagifyClient(_appName, _configPath, _rootPath, _customClientId, _clientStateConfig, environment, _editorDevice);
        }

        /// <inheritdoc cref="MagifyClient.ClearNativeStorage"/>
        internal override void ClearNativeStorage()
        {
            _client.ClearNativeStorage();
        }

        internal override void Setup(TaskScheduler _)
        {
            _client.Setup();
        }

        internal override void Sync()
        {
            _client.Sync();
        }

        internal override void ResetAndForceUpdate()
        {
            _client.ResetContext(true);
            _client.Sync();
        }

        internal override void TweakAnalyticsConfig(int eventsGroupSize, int syncTimeInterval)
        {
            _client.TweakAnalyticsConfig(eventsGroupSize, syncTimeInterval);
        }

        internal override void ResetAnalyticsConfig()
        {
            _client.ResetAnalyticsConfig();
        }

        internal override void TweakFirstLaunchDate(DateTime date)
        {
            _client.TweakFirstLaunchDate(date);
        }

        internal override void TweakUserLocale([NotNull] string languageTag)
        {
            _client.TweakUserLocale(languageTag);
        }

        internal override void SetMediaSource(string networkName = null, string campaignName = null, string adGroup = null)
        {
            _client.SetMediaSource(networkName, campaignName, adGroup);
        }

        internal override void SetAttStatus(bool authorized)
        {
            _client.SetAttStatus(authorized);
        }

        internal override void TrackTrustedPurchase([NotNull] TrustedPurchaseRecord record, bool? isSubscription = default, bool? isExternal = default)
        {
            _client.TrackTrustedPurchase(record, isSubscription, isExternal);
        }

        internal override void SetupConversionTracker(string revenuePerCountryPath, string revenueLevelsPath, string defaultCurrencyRatesPath, string subscriptionMultipliersPath)
        {
            _client.SetupConversionTracker(revenuePerCountryPath, revenueLevelsPath, defaultCurrencyRatesPath, subscriptionMultipliersPath);
        }

        internal override void CompleteOffer(string offerName)
        {
            if (_editorProxy != null)
            {
                _editorProxy.CompleteOffer(offerName);
                return;
            }
            _client.CompleteOffer(offerName);
        }

        internal override ICampaign CampaignFor(string eventName, IReadOnlyDictionary<string, object> customParams = null, bool silent = false)
        {
            if (_editorProxy != null)
            {
                return _editorProxy.CampaignFor(eventName, customParams, silent);
            }
            return _client.CampaignFor(eventName, customParams, silent);
        }

        internal override CampaignImpression LastImpressionFor(CampaignType campaignType, string campaignName, string eventName)
        {
            if (_editorProxy?.IsCampaignFromMoq(campaignType) is true)
            {
                return default;
            }
            return _client.GetCampaignImpression(campaignName);
        }

        internal override bool IsCampaignAvailable(string campaignName, string eventName, Dictionary<string, object> customParams = null)
        {
            return _client.IsCampaignAvailable(campaignName, eventName, customParams);
        }

        internal override void SubscribeCampaignUpdates(string campaignName, Action<ICampaign> onUpdate)
        {
            var campaign = _client.GetLastCampaignRequest(campaignName)?.Campaign;
            if (campaign == null)
            {
                _logger.LogWarning($"Cannot subscribe to updates for campaign {campaignName}. Campaign wasn't requested.");
                return;
            }
            _client.SubscribeCampaignUpdates(campaign, () => onUpdate?.Invoke(campaign));
        }

        internal override void UnsubscribeCampaignUpdates(string campaignName)
        {
            _client.ClearSubscriptions(campaignName);
        }

        internal override bool HasProcessedPurchase(string productId)
        {
            return _client.HasProcessedPurchase(productId);
        }

        internal override void SetGameMode([CanBeNull] string mode)
        {
            _client.SetGameMode(mode);
        }

        internal override void SetGameLevel(int level)
        {
            _client.SetGameLevel(level);
        }

        internal override void SetGameMaxLevel(int maxLevel)
        {
            _client.SetGameMaxLevel(maxLevel);
        }

        internal override void SetUserEmail(string email)
        {
            _client.SetUserEmail(email);
        }

        internal override UniTask<bool> RequestSocialAuthTokenFor([NotNull] string provider, [NotNull] string token, CancellationToken cancellationToken)
        {
            return _client.RequestSocialAuthTokenFor(provider, token, cancellationToken);
        }

        internal override void ResetSocialAuthToken()
        {
            _client.ResetSocialAuthToken();
        }

        internal override UniTask<SyncStateResult> SaveState(int? weight, CancellationToken cancellationToken)
        {
            return _client.SaveState(weight, cancellationToken);
        }

        internal override UniTask<SyncStateResult> RestoreState(int? weight, CancellationToken cancellationToken)
        {
            return _client.RestoreState(weight, cancellationToken);
        }

        internal override void TrackGdprAccessState(bool accessState)
        {
            _client.IsGdprGranted.Value = accessState;
        }

        internal override void TrackAppLaunch()
        {
            // TODO: Remove, called internally from native
        }

        internal override void TrackCustomEvent([NotNull] string eventName, [CanBeNull] IReadOnlyDictionary<string, object> customParams = null)
        {
            _client.TrackCustomEvent(eventName, customParams?.ToDictionary(p => p.Key, p => p.Value));
        }

        internal override void TrackImpression(CampaignType campaignType)
        {
            if (_editorProxy?.IsCampaignFromMoq(campaignType) is true)
            {
                return;
            }
            var campaignName = CampaignTypeToNameOfLastImpressedCampaign(campaignType);
            _client.TrackImpression(campaignName);
        }

        internal override void TrackParentCampaignImpression(CampaignType campaignType)
        {
            if (_editorProxy?.IsCampaignFromMoq(campaignType) is true)
            {
                return;
            }
            var campaignName = CampaignTypeToNameOfLastImpressedCampaign(campaignType);
            _client.TrackParentCampaignImpression(campaignName);
        }

        internal override void TrackProductsImpression(CampaignType campaignType, List<string> productIds)
        {
            if (_editorProxy?.IsCampaignFromMoq(campaignType) is true)
            {
                return;
            }
            var campaignName = CampaignTypeToNameOfLastImpressedCampaign(campaignType);
            _client.TrackProductsImpression(campaignName, productIds);
        }

        internal override void TrackAdsImpression(CampaignType campaignType, IAdsImpression impression)
        {
            if (_editorProxy?.IsCampaignFromMoq(campaignType) is true)
            {
                return;
            }
            var campaignName = CampaignTypeToNameOfLastImpressedCampaign(campaignType);
            _client.TrackAdsImpression(campaignName, impression);
        }

        internal override void TrackImpressionFailFor(CampaignType campaignType, string reason)
        {
            if (_editorProxy?.IsCampaignFromMoq(campaignType) is true)
            {
                return;
            }
            var campaignName = CampaignTypeToNameOfLastImpressedCampaign(campaignType);
            _client.TrackImpressionFailFor(campaignName, reason);
        }

        internal override void TrackClickFor(CampaignType campaignType)
        {
            if (_editorProxy?.IsCampaignFromMoq(campaignType) is true)
            {
                return;
            }
            var campaignName = CampaignTypeToNameOfLastImpressedCampaign(campaignType);
            _client.TrackClickFor(campaignName);
        }

        internal override void TrackAdsClickFor(CampaignType campaignType)
        {
            if (_editorProxy?.IsCampaignFromMoq(campaignType) is true)
            {
                return;
            }
            var campaignName = CampaignTypeToNameOfLastImpressedCampaign(campaignType);
            _client.TrackAdsClickFor(campaignName);
        }

        internal override void TrackProductClickFor(CampaignType campaignType, string productId, PurchaseStore? store)
        {
            if (_editorProxy?.IsCampaignFromMoq(campaignType) is true)
            {
                return;
            }
            var campaignName = CampaignTypeToNameOfLastImpressedCampaign(campaignType);
            _client.TrackProductClickFor(campaignName, productId, store);
        }

        internal override void TrackAdsProductClickFor(CampaignType campaignType, string productId)
        {
            if (_editorProxy?.IsCampaignFromMoq(campaignType) is true)
            {
                return;
            }
            var campaignName = CampaignTypeToNameOfLastImpressedCampaign(campaignType);
            _client.TrackAdsProductClickFor(campaignName, productId);
        }

        [CanBeNull]
        private string CampaignTypeToNameOfLastImpressedCampaign(CampaignType campaignType)
        {
            return _client.GetLastCampaignRequest(campaignType)?.Campaign?.Name;
        }

        internal override bool IsApplicationInstalled([CanBeNull] string identifier)
        {
            return _client.IsApplicationInstalled(identifier);
        }

        internal override void TrackSubscriptionActivation(bool isTrial, string productId, string price, string currency, string transactionId, string purchaseToken, string originalTransactionId, string receipt, bool needValidation, PurchaseStore store)
        {
            _client.TrackSubscriptionActivation(isTrial, productId, price, currency, transactionId, purchaseToken, originalTransactionId, receipt, needValidation, store);
        }

        internal override void TrackExternalSubscriptionActivation(bool isTrial, string productId, string price, string currency, string transactionId, string purchaseToken, string originalTransactionId, string receipt, bool needValidation, PurchaseStore store)
        {
            _client.TrackExternalSubscriptionActivation(isTrial, productId, price, currency, transactionId, purchaseToken, originalTransactionId, receipt, needValidation, store);
        }

        internal override void TrackInAppFor(string productId, string price, string currency, string transactionId, string purchaseToken, string originalTransactionId, string receipt, bool needValidation, PurchaseStore store)
        {
            _client.TrackInAppFor(productId, price, currency, transactionId, purchaseToken, originalTransactionId, receipt, needValidation, store);
        }

        internal override void TrackExternalInAppFor(string productId, string price, string currency, string transactionId, string purchaseToken, string originalTransactionId, string receipt, bool needValidation, PurchaseStore store)
        {
            _client.TrackExternalInAppFor(productId, price, currency, transactionId, purchaseToken, originalTransactionId, receipt, needValidation, store);
        }

        internal override void TrackRestoredInAppFor(string productId)
        {
            _client.TrackRestoredInAppFor(productId);
        }

        internal override void TrackRestoredSubscription([NotNull] string productId)
        {
            _client.TrackRestoredSubscription(productId);
        }

        internal override void TrackIncomeTransaction([NotNull] string source, [NotNull][ItemNotNull] List<BonusInfo> bonuses, [CanBeNull] ProductInfo product = null)
        {
            _client.TrackIncomeTransaction(source, bonuses, product);
        }

        internal override void TrackExpenseTransaction([NotNull] List<BonusInfo> bonuses)
        {
            _client.TrackExpenseTransaction(bonuses);
        }

        internal override void TrackCorrectionTransaction([NotNull] List<BonusInfo> bonuses)
        {
            _client.TrackCorrectionTransaction(bonuses);
        }

        internal override void TrackRewardGranted([NotNull] string productId)
        {
            _client.TrackRewardGranted(productId);
        }

        internal override void TrackFreeBonusGranted([NotNull] string productId)
        {
            _client.TrackFreeBonusGranted(productId);
        }

        internal override void TrackOrdinaryProductUsed([NotNull] string productId)
        {
            _client.TrackOrdinaryProductUsed(productId);
        }

        internal override IList<ContentItem> GetContentList([NotNull] string group, [NotNull] string key, [CanBeNull] List<string> tags = null)
        {
            return _client.GetContentList(group, key, tags);
        }

        internal override ContentItem GetEarliestContent([NotNull] string group, [NotNull] string key)
        {
            return _client.GetEarliestContent(group, key);
        }

        internal override ContentItem GetLatestContent(string group, string key)
        {
            return _client.GetLatestContent(group, key);
        }

        public override void Dispose()
        {
            _client.Dispose();
            _editorProxy?.Dispose();
        }

        #endregion
    }
}