#if false
#pragma warning disable CS0067
using System;
using System.Collections.Generic;
using JetBrains.Annotations;
using Newtonsoft.Json;
using UnityEngine;

namespace Magify
{
    internal class MagifyAndroid : MagifyPlatformAPI, IModernBridgePlatformAPI
    {
        private static readonly AndroidJavaClass _clsManager = new("com.magify.android.plugin.MagifyManager");
        private static readonly AndroidJavaObject _magifyManager = _clsManager.CallStatic<AndroidJavaObject>("getInstance");
        private readonly NativeFeaturesProvider _featuresProvider;

        private static readonly MagifyLogger _logger = MagifyLogger.Get();

        private static event Action OnFeaturesUpdatedFromNative;

        private static event Action<LtoInfo> OnOfferAddedFromNative;
        private static event Action<LtoInfo> OnOfferUpdatedFromNative;
        private static event Action<LtoInfo> OnOfferRemovedFromNative;
        private static event Action<LtoInfo> OnOfferFinishedFromNative;

        private static event Action OnConfigLoadedFromNative;

        private static Dictionary<string, Action<ICampaign>> OnCampaginUpdateCallbacks = new();

        private static event Action<int> OnSessionChangedFromNative;

        private static event Action OnSubscriptionStatusChangedFromNative;
        private static event Action OnInAppStatusChangedFromNative;
        private static event Action OnAuthorizationStatusChangedFromNative;
        private static event Action OnReferrerChangedFromNative;
        private static event Action OnPurchasedProductsChangedFromNative;
        private static event Action<bool> OnForegroundStateChangedFromNative;

        public event Action<LtoInfo> OnOfferAdded;
        public event Action<LtoInfo> OnOfferUpdated;
        public event Action<LtoInfo> OnOfferRemoved;
        public event Action<LtoInfo> OnOfferFinished;

        internal override event Action OnConfigLoaded;

        internal override event Action<int> OnSessionChanged;

        internal override event Action OnSubscriptionStatusChanged;
        internal override event Action OnInAppStatusChanged;
        internal override event Action OnAuthorizationStatusChanged;
        internal override event Action OnReferrerChanged;
        internal override event Action OnPurchasedProductsChanged;

        internal override event Action OnApplicationEnterForeground;
        internal override event Action OnApplicationEnterBackground;

        internal override event Action OnUserDidTakeScreenshot;

        private readonly JsonSerializerSettings _jsonSerializerSettings = new()
        {
            NullValueHandling = NullValueHandling.Ignore
        };

        internal override IReadOnlyCollection<LtoInfo> ActiveLtoOffers => GetActiveLtoOffers();

        internal override Environment Environment
        {
            get => GetEnvironment();
            set => SetEnvironment(value);
        }

        internal override SubscriptionStatus SubscriptionStatus
        {
            get => GetSubscriptionStatus();
            set => SetSubscriptionStatus(value);
        }

        internal override AuthorizationStatus AuthorizationStatus
        {
            get => GetAuthorizationStatus();
            set => SetAuthorizationStatus(value);
        }

        internal override InAppStatus InAppStatus
        {
            get => GetInAppStatus();
            set => throw new NotImplementedException();
        }

        internal override ContextSyncTime LastContextSyncTime => GetLastContextSyncTime();
        internal override IFeatures Features => _featuresProvider;

        internal override bool IsLoggingEnabled
        {
            get => _magifyManager.Call<bool>("isLoggingEnabled");
            set => _magifyManager.Call("setLoggingEnabled", value);
        }

        internal override string ReferrerId
        {
            get => _magifyManager.Call<string>("getReferrerId");
            set => _magifyManager.Call("setReferrerId", value);
        }

        internal override string AdjustId
        {
            get => _magifyManager.Call<string>("getAdjustId");
            set => _magifyManager.Call("setAdjustId", value);
        }

        internal override string FirebaseInstanceId
        {
            get => _magifyManager.Call<string>("getFirebaseAppInstanceId");
            set => _magifyManager.Call("setFirebaseAppInstanceId", value);
        }

        internal override bool IsPortrait => _magifyManager.Call<bool>("isPortraitOrientation");

        internal override Limits Limits => GetLimits();

        internal override string ClientId => _magifyManager.Call<string>("getClientId");

        internal override int SessionNumber => _magifyManager.Call<int>("getSessionNumber");

        internal override List<InAppProduct> InAppProducts => ParseCampaignProducts(_magifyManager.Call<string>("getInAppProducts")).ConvertAll(item => (InAppProduct)item);

        internal override List<SubscriptionProduct> SubscriptionProducts => ParseCampaignProducts(_magifyManager.Call<string>("getSubscriptionProducts")).ConvertAll(item => (SubscriptionProduct)item);

        internal override bool IsSdkInitialized => _magifyManager.Call<bool>("isInitialized");

        internal string DefaultFeatures => _magifyManager.Call<string>("getAllDefaultFeatures");
        internal string CurrentFeatures => _magifyManager.Call<string>("getAllFeatures");

        public MagifyAndroid()
        {
            _featuresProvider = new NativeFeaturesProvider();
        }

        internal override void InitializeSdk(string appName, string configPath, string clientId, Environment environment)
        {
            using var unityPlayer = new AndroidJavaClass("com.unity3d.player.UnityPlayer");
            using var currentActivity = unityPlayer.GetStatic<AndroidJavaObject>("currentActivity");
            using var androidApplication = currentActivity.Call<AndroidJavaObject>("getApplication");

            OnFeaturesUpdatedFromNative += () => { _featuresProvider.UpdateCurrentFeatures(CurrentFeatures); };
            _magifyManager.Call("setOnFeaturesUpdateCallback", new FeaturesUpdateCallback());

            OnOfferAddedFromNative += (info) => { OnOfferAdded?.Invoke(info); };
            _magifyManager.Call("setOfferAddedCallback", new OfferAddedCallback());

            OnOfferUpdatedFromNative += (info) => { OnOfferUpdated?.Invoke(info); };
            _magifyManager.Call("setOfferUpdatedCallback", new OfferUpdatedCallback());

            OnOfferRemovedFromNative += (info) => { OnOfferRemoved?.Invoke(info); };
            _magifyManager.Call("setOfferRemovedCallback", new OfferRemovedCallback());

            OnOfferFinishedFromNative += (info) => { OnOfferFinished?.Invoke(info); };
            _magifyManager.Call("setLtoCompleteCallback", new OfferFinishedCallback());

            OnConfigLoadedFromNative += () => { OnConfigLoaded?.Invoke(); };
            _magifyManager.Call("setConfigLoadedCallback", new ConfigLoadedCallback());

            OnSessionChangedFromNative += (sessionNumber) => { OnSessionChanged?.Invoke(sessionNumber); };
            _magifyManager.Call("setSessionChangeCallback", new SessionChangedCallback());

            OnSubscriptionStatusChangedFromNative += () => { OnSubscriptionStatusChanged?.Invoke(); };
            _magifyManager.Call("setSubscriptionStatusChangeCallback", new SubscriptionStatusChangedCallback());

            OnInAppStatusChangedFromNative += () => { OnInAppStatusChanged?.Invoke(); };
            _magifyManager.Call("setInAppStatusChangeCallback", new InAppStatusChangedCallback());

            OnAuthorizationStatusChangedFromNative += () => { OnAuthorizationStatusChanged?.Invoke(); };
            _magifyManager.Call("setAuthorizationStatusChangeCallback", new AuthorizationStatusChangedCallback());

            OnReferrerChangedFromNative += () => { OnReferrerChanged?.Invoke(); };
            _magifyManager.Call("setReferrerChangeCallback", new ReferrerChangedCallback());

            OnPurchasedProductsChangedFromNative += () => { OnPurchasedProductsChanged?.Invoke(); };
            _magifyManager.Call("setPurchasedProductsChangeCallback", new PurchasedProductsChangedCallback());

            OnForegroundStateChangedFromNative += (isForeground) =>
            {
                if (isForeground) OnApplicationEnterForeground?.Invoke();
                else OnApplicationEnterBackground?.Invoke();
            };
            _magifyManager.Call("setForegroundStateChangeCallback", new ForegroundStateChangedCallback());

            _magifyManager.Call("init", androidApplication, appName, configPath, clientId);
        }

        internal override void Setup(TaskScheduler taskScheduler)
        {
            _magifyManager.Call("setup");
            _featuresProvider.Setup(taskScheduler, DefaultFeatures, CurrentFeatures);
        }

        internal override void Sync()
        {
            _magifyManager.Call("update");
        }

        internal override void Reset()
        {
            _magifyManager.Call("reset");
        }

        internal override void ResetAndForceUpdate()
        {
            _magifyManager.Call("resetAndForceUpdate");
        }

        internal override void TweakAnalyticsConfig(int eventsGroupSize, int syncTimeInterval)
        {
            _magifyManager.Call("tweakAnalyticsConfig", eventsGroupSize, syncTimeInterval);
        }

        internal override void ResetAnalyticsConfig()
        {
            _magifyManager.Call("resetAnalyticsConfig");
        }

        internal override void TweakUserLocale(string languageTag)
        {
            _magifyManager.Call("tweakCurrentLocale", languageTag);
        }

        internal override void SetMediaSource(string networkName = null, string campaignName = null, string adGroup = null)
        {
            _magifyManager.Call("setMediaSource", networkName, campaignName, adGroup);
        }

        internal override void SetAttStatus(bool authorized)
        {
            // Not relevant for Android
        }

        internal override void SetupConversionTracker(string revenuePerCountryPath, string revenueLevelsPath, string defaultCurrencyRatesPath, string subscriptionMultipliersPath)
        {
            // Not relevant for Android
        }

        internal override void CompleteOffer(string offerName)
        {
            _magifyManager.Call("completeOffer", offerName);
        }

        internal override bool IsCampaignAvailable(string campaignName, string eventName, Dictionary<string, object> customParams = null)
        {
            var paramsString = JsonFacade.SerializeObject(customParams, _jsonSerializerSettings);
            return _magifyManager.Call<bool>("isCampaignAvailable", campaignName, eventName, paramsString);
        }

        internal override ICampaign CampaignFor(string eventName, Dictionary<string, object> customParams = null, bool silent = false)
        {
            var paramsString = JsonFacade.SerializeObject(customParams, _jsonSerializerSettings);
            using var campaignObj = _magifyManager.Call<AndroidJavaObject>("provideCampaignFor", eventName, paramsString, silent);
            if (campaignObj == null)
            {
                return null;
            }

            using var typeObj = campaignObj.Call<AndroidJavaObject>("getType");
            var campaignType = (CampaignType)typeObj.Call<int>("getValue");
            var campaignName = campaignObj.Call<string>("getName");

            return GetCampaign(campaignType, campaignName);
        }

        internal override CampaignImpression LastImpressionFor(CampaignType campaignType, string campaignName, string eventName)
        {
            using var impressionObj = _magifyManager.Call<AndroidJavaObject>("getLastImpression", (int)campaignType, campaignName, eventName);
            var attributesDict = Json.Deserialize(impressionObj.Call<string>("getParametersJsonString")) as Dictionary<string, object>;

            return new CampaignImpression
            {
                CampaignName = impressionObj.Call<string>("getCampaignName"),
                ImpressionNumber = impressionObj.Call<int>("getImpressionNumber"),
                SessionImpressionNumber = impressionObj.Call<int>("getSessionImpressionNumber"),
                CampaignTypeImpressionNumber = impressionObj.Call<int>("getCampaignTypeImpressionNumber"),
                CampaignTypeSessionImpressionNumber = impressionObj.Call<int>("getCampaignTypeSessionImpressionNumber"),

                EventName = impressionObj.Call<string>("getEventName"),
                EventNumber = impressionObj.Call<int>("getEventNumber"),
                SessionEventNumber = impressionObj.Call<int>("getSessionEventNumber"),
                Parameters = attributesDict
            };
        }

        internal override void SubscribeCampaignUpdates(string campaignName, Action<ICampaign> onUpdate)
        {
            OnCampaginUpdateCallbacks[campaignName] = onUpdate;
            _magifyManager.Call("subscribeCampaignUpdates", campaignName, new CampaignUpdateCallback());
        }

        internal override void UnsubscribeCampaignUpdates(string campaignName)
        {
            OnCampaginUpdateCallbacks.Remove(campaignName);
            _magifyManager.Call("unsubscribeCampaignUpdates", campaignName);
        }

        internal override bool HasProcessedPurchase(string productId)
        {
            return _magifyManager.Call<bool>("hasProcessedPurchase", productId);
        }

        internal override void SetGameMode(string mode)
        {
            _magifyManager.Call("setGameMode", mode);
        }

        internal override void SetGameLevel(int level)
        {
            _magifyManager.Call("setGameLevel", level);
        }

        internal override void SetGameMaxLevel(int maxLevel)
        {
            _magifyManager.Call("setGameMaxLevel", maxLevel);
        }

        internal override void SetUserEmail(string email)
        {
            _magifyManager.Call("setUserEmail", email);
        }

        internal override void TrackGdprAccessState(bool accessState)
        {
            _magifyManager.Call("setGdprGranted", accessState);
        }

        internal override void TrackAppLaunch()
        {
            // TODO: Remove, called internally from native
        }

        internal override void TrackCustomEvent(string eventName, IReadOnlyDictionary<string, object> customParams = null)
        {
            var paramsString = JsonFacade.SerializeObject(customParams, _jsonSerializerSettings);
            _magifyManager.Call("trackCustomEvent", eventName, paramsString);
        }

        internal override void TrackClickFor(CampaignType campaignType)
        {
            _magifyManager.Call("trackClick", (int)campaignType);
        }

        internal override void TrackAdsClickFor(CampaignType campaignType)
        {
            _magifyManager.Call("trackAdsClick", (int)campaignType);
        }

        internal override void TrackProductClickFor(CampaignType campaignType, string productId)
        {
            _magifyManager.Call("trackProductClick", (int)campaignType, productId);
        }

        internal override void TrackAdsProductClickFor(CampaignType campaignType, string productId)
        {
            _magifyManager.Call("trackAdsProductClick", (int)campaignType, productId);
        }

        internal override void TrackImpression(CampaignType campaignType)
        {
            _magifyManager.Call("trackImpression", (int)campaignType);
        }

        internal override void TrackParentCampaignImpression(CampaignType campaignType)
        {
            _magifyManager.Call("trackParentCampaignImpression", (int)campaignType);
        }

        internal override void TrackProductsImpression(CampaignType campaignType, List<string> productIds)
        {
            var productsString = JsonFacade.SerializeObject(productIds, _jsonSerializerSettings);
            _magifyManager.Call("trackProductsImpression", (int)campaignType, productsString);
        }

        internal override void TrackAdsImpression(CampaignType campaignType, IAdsImpression impression)
        {
            var impressionString = JsonFacade.SerializeObject(impression, _jsonSerializerSettings);
            _magifyManager.Call("trackAdsImpression", (int)campaignType, (int)impression.Mediation, impressionString);
        }

        internal override void TrackImpressionFailFor(CampaignType campaignType, string reason)
        {
            _magifyManager.Call("trackImpressionFail", (int)campaignType, reason);
        }

        internal override void TrackInAppFor(string productId, string price, string currency, string transactionId, string purchaseToken, string originalTransactionId, string receipt)
        {
            _magifyManager.Call("trackInApp", productId, price, currency, transactionId, purchaseToken);
        }

        internal override void TrackExternalInAppFor(string productId, string price, string currency, string transactionId, string purchaseToken, string originalTransactionId, string receipt)
        {
            _magifyManager.Call("trackExternalInApp", productId, price, currency, transactionId, purchaseToken);
        }

        internal override void TrackRestoredInAppFor(string productId)
        {
        }

        internal override void TrackSubscriptionActivation(bool isTrial, string productId, string price, string currency, string period, string transactionId, string purchaseToken, string originalTransactionId, string receipt)
        {
            _magifyManager.Call("trackSubscriptionActivation", isTrial, productId, price, currency, transactionId, purchaseToken);
        }

        internal override void TrackExternalSubscriptionActivation(bool isTrial, string productId, string price, string currency, string transactionId, string purchaseToken, string originalTransactionId, string receipt)
        {
            _magifyManager.Call("trackExternalSubscriptionActivation", isTrial, productId, price, currency, transactionId, purchaseToken);
        }

        internal override void TrackIncomeTransaction(string source, List<BonusInfo> bonuses, ProductInfo product = null)
        {
            _magifyManager.Call("trackIncomeTransaction", source, JsonFacade.SerializeObject(bonuses, _jsonSerializerSettings), product?.ProductId, product?.Price, product?.Currency);
        }

        internal override void TrackExpenseTransaction(List<BonusInfo> bonuses)
        {
            _magifyManager.Call("trackExpenseTransaction", JsonFacade.SerializeObject(bonuses, _jsonSerializerSettings));
        }

        internal override void TrackCorrectionTransaction(List<BonusInfo> bonuses)
        {
            _logger.LogError($"Correction transactions are not supported in current version");
        }

        internal override void TrackRewardGranted(string productId)
        {
            _magifyManager.Call("trackRewardGranted", productId);
        }

        internal override void TrackFreeBonusGranted(string productId)
        {
            _magifyManager.Call("trackFreeBonusGranted", productId);
        }

        internal override void TrackOrdinaryProductUsed(string productId)
        {
            _magifyManager.Call("trackOrdinaryProductUsed", productId);
        }

        internal override List<ContentItem> GetContentList(string group, string key, List<string> tags = null)
        {
            var contentListString = _magifyManager.Call<string>("getContentList", group, key, JsonFacade.SerializeObject(tags, _jsonSerializerSettings));

            var rawContentList = (List<object>)Json.Deserialize(contentListString);
            var resultItems = new List<ContentItem>();
            foreach (var contentItem in rawContentList)
            {
                var contentItemDict = (Dictionary<string, object>)contentItem;
                var itemKey = contentItemDict["key"] as string;
                var itemLink = contentItemDict["link"] as string;
                var itemPreviews = ((List<object>)contentItemDict["previews"]).ConvertAll(x => x.ToString());
                var itemTags = ((List<object>)contentItemDict["tags"]).ConvertAll(x => x.ToString());

                var item = new ContentItem
                {
                    Key = itemKey,
                    Link = itemLink,
                    Previews = itemPreviews,
                    Tags = itemTags
                };
                resultItems.Add(item);
            }

            return resultItems;
        }

        internal override ContentItem GetEarliestContent(string group, string key)
        {
            return GetContentItem("getEarliestContent", group, key);
        }

        internal override ContentItem GetLatestContent(string group, string key)
        {
            return GetContentItem("getLatestContent", group, key);
        }

        public override void Dispose()
        {
            // ToDo: try to free all resources and subscriptions
        }

        // --

        private static ContentItem GetContentItem(string methodName, string group, string key)
        {
            using var contentObj = _magifyManager.Call<AndroidJavaObject>(methodName, group, key);
            if (contentObj == null)
            {
                return null;
            }

            var rawPreviews = (List<object>)Json.Deserialize(contentObj.Call<string>("getPreviews"));
            var previews = rawPreviews.ConvertAll(x => x.ToString());

            var rawTags = (List<object>)Json.Deserialize(contentObj.Call<string>("getTags"));
            var tags = rawTags.ConvertAll(x => x.ToString());

            return new ContentItem
            {
                Key = contentObj.Call<string>("getKey"),
                Link = contentObj.Call<string>("getLink"),
                Previews = previews,
                Tags = tags
            };
        }

        private static Environment GetEnvironment()
        {
            using var envClass = new AndroidJavaClass("com.magify.sdk.network.Environment");
            using var envObj = _magifyManager.Call<AndroidJavaObject>("getEnvironment");

            // ReSharper disable once NotResolvedInText
            if (envObj == null) throw new ArgumentNullException("Environment");

            var envString = envObj.Call<string>("toString");
            using var prodEnvObj = envClass.GetStatic<AndroidJavaObject>("PROD");
            if (envString == prodEnvObj.Call<string>("toString"))
            {
                return Environment.Production;
            }

            using var stageEnvObj = envClass.GetStatic<AndroidJavaObject>("STAGE");
            if (envString == stageEnvObj.Call<string>("toString"))
            {
                return Environment.Staging;
            }

            throw new ArgumentException("Invalid environment value: " + envString);
        }

        private static void SetEnvironment(Environment environment)
        {
            using var envClass = new AndroidJavaClass("com.magify.sdk.network.Environment");
            using var envObject = environment switch
            {
                Environment.Production => envClass.GetStatic<AndroidJavaObject>("PROD"),
                Environment.Staging => envClass.GetStatic<AndroidJavaObject>("STAGE"),
                _ => null
            };

            _magifyManager.Call("setEnvironment", envObject);
        }

        private static SubscriptionStatus GetSubscriptionStatus()
        {
            using var statusClass = new AndroidJavaClass("com.magify.sdk.model.config.SubscriptionState");
            using var statusObj = _magifyManager.Call<AndroidJavaObject>("getSubscriptionStatus");

            // ReSharper disable once NotResolvedInText
            if (statusObj == null) throw new ArgumentNullException("SubscriptionStatus");

            var status = statusObj.Call<string>("toString");
            using var inactiveObj = statusClass.GetStatic<AndroidJavaObject>("INACTIVE");
            if (status == inactiveObj.Call<string>("toString"))
            {
                return SubscriptionStatus.Inactive;
            }

            using var activeTrialObj = statusClass.GetStatic<AndroidJavaObject>("ACTIVE_TRIAL");
            if (status == activeTrialObj.Call<string>("toString"))
            {
                return SubscriptionStatus.Trial;
            }

            using var activePaidObj = statusClass.GetStatic<AndroidJavaObject>("ACTIVE_PAID");
            if (status == activePaidObj.Call<string>("toString"))
            {
                return SubscriptionStatus.Paid;
            }

            using var expiredTrialObj = statusClass.GetStatic<AndroidJavaObject>("EXPIRED_TRIAL");
            if (status == expiredTrialObj.Call<string>("toString"))
            {
                return SubscriptionStatus.ExpiredTrial;
            }

            using var expiredPaidObj = statusClass.GetStatic<AndroidJavaObject>("EXPIRED_PAID");
            if (status == expiredPaidObj.Call<string>("toString"))
            {
                return SubscriptionStatus.ExpiredPaid;
            }

            throw new ArgumentException("Invalid subscriptionStatus value: " + status);
        }

        private static void SetSubscriptionStatus(SubscriptionStatus status)
        {
            using var statusClass = new AndroidJavaClass("com.magify.sdk.model.config.SubscriptionState");
            using var statusObject = status switch
            {
                SubscriptionStatus.Inactive => statusClass.GetStatic<AndroidJavaObject>("INACTIVE"),
                SubscriptionStatus.Trial => statusClass.GetStatic<AndroidJavaObject>("ACTIVE_TRIAL"),
                SubscriptionStatus.Paid => statusClass.GetStatic<AndroidJavaObject>("ACTIVE_PAID"),
                SubscriptionStatus.ExpiredTrial => statusClass.GetStatic<AndroidJavaObject>("EXPIRED_TRIAL"),
                SubscriptionStatus.ExpiredPaid => statusClass.GetStatic<AndroidJavaObject>("EXPIRED_PAID"),
                _ => null
            };

            _magifyManager.Call("setSubscriptionStatus", statusObject);
        }

        private static AuthorizationStatus GetAuthorizationStatus()
        {
            using var statusClass = new AndroidJavaClass("com.magify.sdk.model.config.AuthorizationStatus");
            using var statusObj = _magifyManager.Call<AndroidJavaObject>("getAuthorizationStatus");

            // ReSharper disable once NotResolvedInText
            if (statusObj == null) throw new ArgumentNullException("AuthorizationStatus");

            var status = statusObj.Call<string>("toString");
            using var authorizedObj = statusClass.GetStatic<AndroidJavaObject>("AUTHORIZED");
            if (status == authorizedObj.Call<string>("toString"))
            {
                return AuthorizationStatus.Authorized;
            }

            using var unauthorizedObj = statusClass.GetStatic<AndroidJavaObject>("UNAUTHORIZED");
            if (status == unauthorizedObj.Call<string>("toString"))
            {
                return AuthorizationStatus.Unauthorized;
            }

            using var unknownObj = statusClass.GetStatic<AndroidJavaObject>("UNKNOWN");
            if (status == unknownObj.Call<string>("toString"))
            {
                return AuthorizationStatus.Unknown;
            }

            throw new ArgumentException("Invalid AuthorizationStatus value: " + status);
        }

        private static void SetAuthorizationStatus(AuthorizationStatus status)
        {
            using var statusClass = new AndroidJavaClass("com.magify.sdk.model.config.AuthorizationStatus");
            using var statusObject = status switch
            {
                AuthorizationStatus.Authorized => statusClass.GetStatic<AndroidJavaObject>("AUTHORIZED"),
                AuthorizationStatus.Unauthorized => statusClass.GetStatic<AndroidJavaObject>("UNAUTHORIZED"),
                AuthorizationStatus.Unknown => statusClass.GetStatic<AndroidJavaObject>("UNKNOWN"),
                _ => null
            };

            _magifyManager.Call("setAuthorizationStatus", statusObject);
        }

        private static InAppStatus GetInAppStatus()
        {
            using var statusClass = new AndroidJavaClass("com.magify.sdk.model.config.InAppStatus");
            using var statusObj = _magifyManager.Call<AndroidJavaObject>("getInAppStatus");

            // ReSharper disable once NotResolvedInText
            if (statusObj == null) throw new ArgumentNullException("InAppStatus");

            var status = statusObj.Call<string>("toString");
            using var neverPurchasedObj = statusClass.GetStatic<AndroidJavaObject>("NEVER_PURCHASED");
            if (status == neverPurchasedObj.Call<string>("toString"))
            {
                return InAppStatus.NeverPurchased;
            }

            using var purchasedObj = statusClass.GetStatic<AndroidJavaObject>("PURCHASED");
            if (status == purchasedObj.Call<string>("toString"))
            {
                return InAppStatus.Purchased;
            }

            throw new ArgumentException("Invalid InAppStatus value: " + status);
        }

        private static Limits GetLimits()
        {
            using var limitsObj = _magifyManager.Call<AndroidJavaObject>("getLimits");

            Limits limits;
            limits.GlobalInterval = limitsObj.Call<double>("getGlobalInterval");
            limits.SessionsInterval = limitsObj.Call<double>("getSessionsInterval");
            limits.InterstitialInterval = limitsObj.Call<double>("getInterstitialInterval");
            limits.RewardInterstitialInterval = limitsObj.Call<double>("getRewardInterstitialInterval");
            limits.InterstitialsPerSession = limitsObj.Call<int>("getInterstitialPerSession");
            limits.ImpressionsPerSession = limitsObj.Call<int>("getImpressionPerSession");

            return limits;
        }

        private static ContextSyncTime GetLastContextSyncTime()
        {
            using var timeObj = _magifyManager.Call<AndroidJavaObject>("getContextSyncTime");
            if (timeObj == null)
            {
                return null;
            }

            return new ContextSyncTime
            {
                LocalTimeMillis = timeObj.Call<long>("getLocalTimeMillis"),
                RemoteTimeMillis = timeObj.Call<long>("getRemoteTimeMillis")
            };
        }

        private static ICampaign GetCampaign(CampaignType campaignType, string campaignName)
        {
            switch (campaignType)
            {
                case CampaignType.Banner:
                    return GetBannerCampaign(campaignName);

                case CampaignType.RewardedVideo:
                case CampaignType.LtoRewardedVideo:
                    return GetRewardedVideoCampaign(campaignType, campaignName);

                case CampaignType.RateReview:
                    var reviewCampaign = new RateReviewCampaign
                    {
                        Name = campaignName
                    };
                    return reviewCampaign;

                case CampaignType.Interstitial:
                    return GetInterstitialCampaign(campaignName);

                case CampaignType.Subscription:
                case CampaignType.LtoSubscription:
                    return GetSubscriptionCampaign(campaignType, campaignName);

                case CampaignType.CrossPromo:
                case CampaignType.LtoCrossPromo:
                case CampaignType.External:
                case CampaignType.LtoExternal:
                    return GetPromoCampaign(campaignType, campaignName);

                case CampaignType.InApp:
                case CampaignType.LtoInApp:
                    return GetInAppCampaign(campaignType, campaignName);

                case CampaignType.Notification:
                    return GetNotificationCampaign(campaignName);

                case CampaignType.Bonus:
                case CampaignType.LtoBonus:
                    return GetBonusCampaign(campaignType, campaignName);

                case CampaignType.Mixed:
                case CampaignType.LtoMixed:
                    return GetMixedCampaign(campaignType, campaignName);

                default:
                    return null;
            }
        }

        private static BannerCampaign GetBannerCampaign(string name)
        {
            using var campaignObj = _magifyManager.Call<AndroidJavaObject>("getCurrentBannerCampaign");
            if (campaignObj == null)
            {
                return null;
            }

            using var positionObj = campaignObj.Call<AndroidJavaObject>("getPosition");
            return new BannerCampaign
            {
                Name = name,
                Position = (BannerPosition)positionObj.Call<int>("getValue"),
            };
        }

        private static InterstitialCampaign GetInterstitialCampaign(string name)
        {
            using var campaignObj = _magifyManager.Call<AndroidJavaObject>("getCurrentInterstitialCampaign");
            if (campaignObj == null)
            {
                return null;
            }

            using var splashObj = campaignObj.Call<AndroidJavaObject>("getSplashscreen");
            using var splashTypeObj = splashObj.Call<AndroidJavaObject>("getType");
            var attributesDict = Json.Deserialize(splashObj.Call<string>("getAttributes")) as Dictionary<string, object>;
            var splash = new InterScreen
            {
                Type = (SplashscreenType)splashTypeObj.Call<int>("getValue"),
                MinTime = splashObj.Call<double>("getMinTime"),
                MaxTime = splashObj.Call<double>("getMaxTime"),
                ExactTime = splashObj.Call<double>("getExactTime"),
                Timeout = splashObj.Call<double>("getTimeout"),
                IsForced = splashObj.Call<bool>("isForced"),
                Attributes = attributesDict
            };

            return new InterstitialCampaign
            {
                Name = name,
                Screen = splash.ToSplashScreen()
            };
        }

        private static SubscriptionCampaign GetSubscriptionCampaign(CampaignType type, string name)
        {
            using var campaignObj = _magifyManager.Call<AndroidJavaObject>("getCurrentSubscriptionCampaign", (int)type);
            if (campaignObj == null)
            {
                return null;
            }

            var products = ParseCampaignProducts(campaignObj.Call<string>("getProducts"));
            using var screenObj = campaignObj.Call<AndroidJavaObject>("getScreen");

            return new SubscriptionCampaign
            {
                Name = name,
                Type = type,
                Creative = GetCampaignScreen(screenObj).ToCreative(),
                Products = products.ConvertAll(item => (SubscriptionProduct)item)
            };
        }

        private static ICampaign GetPromoCampaign(CampaignType type, string name)
        {
            var objectMethod = type is CampaignType.CrossPromo or CampaignType.LtoCrossPromo
                ? "getCurrentCrossPromoCampaign"
                : "getCurrentExternalCampaign";

            using var campaignObj = _magifyManager.Call<AndroidJavaObject>(objectMethod, (int)type);
            if (campaignObj == null)
            {
                return null;
            }

            var destinationObj = campaignObj.Call<AndroidJavaObject>("getDestination");
            var screenObj = campaignObj.Call<AndroidJavaObject>("getScreen");

            var creative = GetCampaignScreen(screenObj).ToCreative();
            var destination = (DestinationType)destinationObj.Call<AndroidJavaObject>("getType").Call<int>("getValue");
            var url = destinationObj.Call<string>("getValue");

            return destination switch
            {
                DestinationType.Application => new CrossPromoCampaign
                {
                    Type = type,
                    Creative = creative,
                    Name = name,
                    Products = new List<CrossPromoProduct>
                    {
                        new(ProductDef.FakeProductId, destinationObj.Call<string>("getPackageName"), url, null, null, creative.Attributes)
                    }
                },
                DestinationType.Link => new ExternalPromoCampaign
                {
                    Type = type,
                    Creative = creative,
                    Name = name,
                    Products = new List<ExternalLinkProduct>
                    {
                        new(ProductDef.FakeProductId, url, null, creative.Attributes)
                    }
                },
                _ => null
            };
        }

        private static InAppCampaign GetInAppCampaign(CampaignType type, string name)
        {
            using var campaignObj = _magifyManager.Call<AndroidJavaObject>("getCurrentInAppCampaign", (int)type);
            if (campaignObj == null)
            {
                return null;
            }

            var products = ParseCampaignProducts(campaignObj.Call<string>("getProducts"));
            using var screenObj = campaignObj.Call<AndroidJavaObject>("getScreen");

            return new InAppCampaign
            {
                Name = name,
                Type = type,
                Creative = GetCampaignScreen(screenObj).ToCreative(),
                Products = products.ConvertAll(item => (InAppProduct)item)
            };
        }

        private static RewardedVideoCampaign GetRewardedVideoCampaign(CampaignType type, string name)
        {
            using var campaignObj = _magifyManager.Call<AndroidJavaObject>("getCurrentRewardedCampaign", (int)type);
            if (campaignObj == null)
            {
                return null;
            }

            var products = ParseCampaignProducts(campaignObj.Call<string>("getProducts"));
            using var screenObj = campaignObj.Call<AndroidJavaObject>("getScreen");

            return new RewardedVideoCampaign
            {
                Name = name,
                Type = type,
                Creative = GetCampaignScreen(screenObj).ToCreative(),
                Products = products.ConvertAll(item => (RewardProduct)item)
            };
        }

        private static CampaignScreen GetCampaignScreen(AndroidJavaObject screenObj)
        {
            var attributesDict = Json.Deserialize(screenObj.Call<string>("getAttributes")) as Dictionary<string, object>;
            using var screenTypeObj = screenObj.Call<AndroidJavaObject>("getType");
            return new CampaignScreen
            {
                Type = (ScreenType)screenTypeObj.Call<int>("getValue"),
                ScreenId = screenObj.Call<string>("getScreenId"),
                Attributes = attributesDict,
                PortraitUrl = screenObj.Call<string>("getPortraitUrl"),
                LandscapeUrl = screenObj.Call<string>("getLandscapeUrl"),
                ButtonCloseTimeout = screenObj.Call<float>("getButtonCloseTimeout"),
                Color = screenObj.Call<string>("getBgColor").ToColor(),
                Opacity = screenObj.Call<int>("getBgOpacity"),
                BlurBackground = screenObj.Call<bool>("getBgBlurred")
            };
        }

        private static NotificationCampaign GetNotificationCampaign(string name)
        {
            using var campaignObj = _magifyManager.Call<AndroidJavaObject>("getCurrentNotificationCampaign");
            if (campaignObj == null)
            {
                return null;
            }

            using var screenObj = campaignObj.Call<AndroidJavaObject>("getScreen");
            return new NotificationCampaign
            {
                Name = name,
                Creative = GetCampaignScreen(screenObj).ToCreative()
            };
        }

        private static BonusCampaign GetBonusCampaign(CampaignType type, string name)
        {
            using var campaignObj = _magifyManager.Call<AndroidJavaObject>("getCurrentBonusCampaign", (int)type);
            if (campaignObj == null)
            {
                return null;
            }

            var products = ParseCampaignProducts(campaignObj.Call<string>("getProducts"));
            using var screenObj = campaignObj.Call<AndroidJavaObject>("getScreen");

            return new BonusCampaign
            {
                Name = name,
                Type = type,
                Creative = GetCampaignScreen(screenObj).ToCreative(),
                Products = products.ConvertAll(item => (BonusProduct)item)
            };
        }

        private static MixedCampaign GetMixedCampaign(CampaignType type, string name)
        {
            using var campaignObj = _magifyManager.Call<AndroidJavaObject>("getCurrentMixedCampaign", (int)type);
            if (campaignObj == null)
            {
                return null;
            }

            var products = ParseCampaignProducts(campaignObj.Call<string>("getProducts"));
            using var screenObj = campaignObj.Call<AndroidJavaObject>("getScreen");

            return new MixedCampaign
            {
                Name = name,
                Type = type,
                Creative = GetCampaignScreen(screenObj).ToCreative(),
                Products = products
            };
        }

        private static List<ProductDef> ParseCampaignProducts(string productsString)
        {
            var productsList = (List<object>)Json.Deserialize(productsString);
            var products = new List<ProductDef>();
            foreach (var product in productsList)
            {
                var productDict = (Dictionary<string, object>)product;
                var productId = (string)productDict["productId"];
                var context = productDict["productContext"] as Dictionary<string, object>;
                var productType = (ProductType)int.Parse(productDict["productType"].ToString());
                string link = string.Empty;
                if (productDict.TryGetValue("link", out var linkObject))
                {
                    link = (string)linkObject;
                }

                string promoBundleId = null;
                if (productDict.TryGetValue("promoAppPackageName", out var bundleObject))
                {
                    promoBundleId = (string)bundleObject;
                }

                string promoUrl = null;
                if (productDict.TryGetValue("promoAppUrl", out var promoUrlObject))
                {
                    promoUrl = (string)promoUrlObject;
                }

                ProductCreative creative = null;
                if (productDict.TryGetValue("cardCreative", out var creativeObject))
                {
                    creative = ParseCardCreative((Dictionary<string, object>)creativeObject);
                }

                ProductDef productItem = productType switch
                {
                    ProductType.Rewarded => new RewardProduct(productId, int.Parse(productDict["count"].ToString()), creative, context),
                    ProductType.Subscription => new SubscriptionProduct(productId, creative, context),
                    ProductType.Bonus => new BonusProduct(productId, creative, context),
                    ProductType.Info => new InfoProduct(productId, creative, context),
                    ProductType.ExternalLink => new ExternalLinkProduct(productId, link, creative, context),
                    ProductType.InternalLink => new InternalLinkProduct(productId, link, creative, context),
                    ProductType.Consumable => new InAppProduct(productId, true, creative, context),
                    ProductType.NonConsumable => new InAppProduct(productId, false, creative, context),
                    ProductType.CrossPromo => new CrossPromoProduct(productId, promoBundleId, promoUrl, new List<string>(), creative, context),
                    _ => throw new ArgumentException("Invalid product type: " + productType)
                };
                products.Add(productItem);
            }

            return products;
        }

        private static IReadOnlyCollection<LtoInfo> GetActiveLtoOffers()
        {
            var offersString = _magifyManager.Call<string>("getActiveLtoOffers");
            return ParseLtoItemsList(offersString);
        }

        private static IReadOnlyCollection<LtoInfo> ParseLtoItemsList(string offersString)
        {
            var itemsList = (List<object>)Json.Deserialize(offersString);
            var offers = new List<LtoInfo>();
            foreach (var item in itemsList)
            {
                var offerDict = (Dictionary<string, object>)item;
                var info = ParseLtoItem(offerDict);
                offers.Add(info);
            }

            return offers;
        }

        private static LtoInfo ParseLtoItem(IReadOnlyDictionary<string, object> offerDict)
        {
            return new LtoInfo
            {
                CampaignName = (string)offerDict["name"],
                Spot = (string)offerDict["spot"],
                StartTime = UnixTimestamp.ToDateTime(int.Parse(offerDict["startTimeSec"].ToString())),
                EndTime = UnixTimestamp.ToDateTime(int.Parse(offerDict["endTimeSec"].ToString())),
                BadgeCreative = ParseBadgeCreative((Dictionary<string, object>)offerDict["badgeCreative"])
            };
        }

        private static LtoBadgeCreative ParseBadgeCreative(IReadOnlyDictionary<string, object> badgeDict)
        {
            var attributesDict = Json.Deserialize((string)badgeDict["attributes"]) as Dictionary<string, object>;
            return new LtoBadgeCreative
            {
                Tag = (string)badgeDict["tag"],
                Title = (string)badgeDict["title"],
                LabelText = (string)badgeDict["labelText"],
                Resource = ParseCreativeResource((Dictionary<string, object>)badgeDict["resource"]),
                Placeholder = (string)badgeDict["placeholder"],
                HasTimer = bool.Parse(badgeDict["hasTimer"].ToString()),
                LastMinutesTimer = int.Parse(badgeDict["lastMinutesTimer"].ToString()),
                Attributes = attributesDict
            };
        }

        private static ProductCreative ParseCardCreative(IReadOnlyDictionary<string, object> dictionary)
        {
            var attributesDict = Json.Deserialize((string)dictionary["attributes"]) as Dictionary<string, object>;
            return new ProductCreative
            {
                Title = (string)dictionary["title"],
                Text = (string)dictionary["labelText"],
                Resource = ParseCreativeResource((Dictionary<string, object>)dictionary["resource"]),
                ActionButton = (string)dictionary["actionButton"],
                CloseButton = (string)dictionary["closeButton"],
                Attributes = attributesDict
            };
        }

        private static CreativeResource ParseCreativeResource(IReadOnlyDictionary<string, object> resourceDict)
        {
            var type = (CreativeResource.ArtifactType)int.Parse(resourceDict["type"].ToString());
            var url = (string)resourceDict["url"];

            return new CreativeResource
            {
                Url = url,
                Type = type
            };
        }

        private class FeaturesUpdateCallback : AndroidJavaProxy
        {
            public FeaturesUpdateCallback() : base("com.magify.android.plugin.OnFeaturesUpdateCallback")
            {
            }

            [UsedImplicitly]
            public void onUpdate()
            {
                OnFeaturesUpdatedFromNative?.Invoke();
            }
        }

        private class OfferAddedCallback : AndroidJavaProxy
        {
            public OfferAddedCallback() : base("com.magify.android.plugin.OnOfferAddedCallback")
            {
            }

            [UsedImplicitly]
            public void onAdded(string offerString)
            {
                var offerDict = (Dictionary<string, object>)Json.Deserialize(offerString);
                var offer = ParseLtoItem(offerDict);
                OnOfferAddedFromNative?.Invoke(offer);
            }
        }

        private class OfferUpdatedCallback : AndroidJavaProxy
        {
            public OfferUpdatedCallback() : base("com.magify.android.plugin.OnOfferUpdatedCallback")
            {
            }

            [UsedImplicitly]
            public void onUpdated(string offerString)
            {
                var offerDict = (Dictionary<string, object>)Json.Deserialize(offerString);
                var offer = ParseLtoItem(offerDict);
                OnOfferUpdatedFromNative?.Invoke(offer);
            }
        }

        private class OfferRemovedCallback : AndroidJavaProxy
        {
            public OfferRemovedCallback() : base("com.magify.android.plugin.OnOfferRemovedCallback")
            {
            }

            [UsedImplicitly]
            public void onRemoved(string offerString)
            {
                var offerDict = (Dictionary<string, object>)Json.Deserialize(offerString);
                var offer = ParseLtoItem(offerDict);
                OnOfferRemovedFromNative?.Invoke(offer);
            }
        }

        private class OfferFinishedCallback : AndroidJavaProxy
        {
            public OfferFinishedCallback() : base("com.magify.android.plugin.OnLtoCompleteCallback")
            {
            }

            [UsedImplicitly]
            public void onComplete(string offerString)
            {
                var offerDict = (Dictionary<string, object>)Json.Deserialize(offerString);
                var offer = ParseLtoItem(offerDict);
                OnOfferFinishedFromNative?.Invoke(offer);
            }
        }

        private class ConfigLoadedCallback : AndroidJavaProxy
        {
            public ConfigLoadedCallback() : base("com.magify.android.plugin.OnConfigLoadedCallback")
            {
            }

            [UsedImplicitly]
            public void onLoaded()
            {
                OnConfigLoadedFromNative?.Invoke();
            }
        }

        private class CampaignUpdateCallback : AndroidJavaProxy
        {
            public CampaignUpdateCallback() : base("com.magify.android.plugin.OnCampaignUpdateCallback")
            {
            }

            [UsedImplicitly]
            public void onUpdate(int campaignType, string campaignName)
            {
                if (!OnCampaginUpdateCallbacks.ContainsKey(campaignName)) return;

                var campaign = GetCampaign((CampaignType)campaignType, campaignName);
                OnCampaginUpdateCallbacks[campaignName]?.Invoke(campaign);
            }
        }

        private class SessionChangedCallback : AndroidJavaProxy
        {
            public SessionChangedCallback() : base("com.magify.android.plugin.callbacks.OnSessionChangeCallback")
            {
            }

            [UsedImplicitly]
            public void onChange(int sessionNumber)
            {
                OnSessionChangedFromNative?.Invoke(sessionNumber);
            }
        }

        private class SubscriptionStatusChangedCallback : AndroidJavaProxy
        {
            public SubscriptionStatusChangedCallback() : base("com.magify.android.plugin.callbacks.OnSubscriptionStatusChangeCallback")
            {
            }

            [UsedImplicitly]
            public void onChange()
            {
                OnSubscriptionStatusChangedFromNative?.Invoke();
            }
        }

        private class InAppStatusChangedCallback : AndroidJavaProxy
        {
            public InAppStatusChangedCallback() : base("com.magify.android.plugin.callbacks.OnInAppStatusChangeCallback")
            {
            }

            [UsedImplicitly]
            public void onChange()
            {
                OnInAppStatusChangedFromNative?.Invoke();
            }
        }

        private class AuthorizationStatusChangedCallback : AndroidJavaProxy
        {
            public AuthorizationStatusChangedCallback() : base("com.magify.android.plugin.callbacks.OnAuthorizationStatusChangeCallback")
            {
            }

            [UsedImplicitly]
            public void onChange()
            {
                OnAuthorizationStatusChangedFromNative?.Invoke();
            }
        }

        private class ReferrerChangedCallback : AndroidJavaProxy
        {
            public ReferrerChangedCallback() : base("com.magify.android.plugin.callbacks.OnReferrerChangeCallback")
            {
            }

            [UsedImplicitly]
            public void onChange()
            {
                OnReferrerChangedFromNative?.Invoke();
            }
        }

        private class PurchasedProductsChangedCallback : AndroidJavaProxy
        {
            public PurchasedProductsChangedCallback() : base("com.magify.android.plugin.callbacks.OnPurchasedProductsChangeCallback")
            {
            }

            [UsedImplicitly]
            public void onChange()
            {
                OnPurchasedProductsChangedFromNative?.Invoke();
            }
        }

        private class ForegroundStateChangedCallback : AndroidJavaProxy
        {
            public ForegroundStateChangedCallback() : base("com.magify.android.plugin.OnForegroundStateChangeCallback")
            {
            }

            [UsedImplicitly]
            public void onChange(bool isForeground)
            {
                OnForegroundStateChangedFromNative?.Invoke(isForeground);
            }
        }
    }
}
#endif