using System;
using JetBrains.Annotations;
using Newtonsoft.Json;
using UnityEngine.Scripting;

namespace Magify
{
    [Preserve]
    public class LtoInfo
    {
        /// <summary>
        /// The name of the campaign running by this LTO
        /// </summary>
        [Preserve]
        [JsonProperty("campaign_name")]
        public string CampaignName { get; internal set; }

        /// <summary>
        /// Spot position (anchor) of the LTO campaign
        /// </summary>
        [Preserve]
        [JsonProperty("spot")]
        public string Spot { get; init; }

        /// <summary>
        /// UTC-0 Timestamp in seconds comes from the server and it converted to DateTime
        /// </summary>
        [Preserve]
        [JsonProperty("start_time")]
        public DateTime StartTime { get; internal set; }

        /// <summary>
        /// UTC-0 Timestamp in seconds comes from the server and it converted to DateTime
        /// </summary>
        [Preserve]
        [JsonProperty("end_time")]
        public DateTime EndTime { get; internal set; }

        /// <summary>
        /// TimeSpan to the end of the offer.
        /// </summary>
        [JsonIgnore]
        public TimeSpan RemainingTime => EndTime - DateTime.UtcNow;

        [JsonIgnore]
        public LtoBadgeCreative BadgeCreative { get; internal set; }

#if UNITY_EDITOR
        [NotNull]
        public static LtoInfo Moq(
            [CanBeNull] string campaignName,
            [CanBeNull] string spot,
            DateTime startTime,
            DateTime endTime,
            [CanBeNull] LtoBadgeCreative badgeCreative)
        {
            return new LtoInfo
            {
                CampaignName = campaignName,
                Spot = spot,
                StartTime = startTime,
                EndTime = endTime,
                BadgeCreative = badgeCreative,
            };
        }
#endif

        internal void TakeUpdatesFrom(LtoInfo other)
        {
            EndTime = other.EndTime;
            BadgeCreative = other.BadgeCreative;
        }

        public bool Equals(LtoInfo other)
        {
            return CampaignName == other.CampaignName &&
                   Spot == other.Spot &&
                   StartTime == other.StartTime &&
                   EndTime == other.EndTime &&
                   BadgeCreative.Equals(other.BadgeCreative);
        }

        public override bool Equals(object obj)
        {
            if (ReferenceEquals(null, obj)) return false;
            if (ReferenceEquals(this, obj)) return true;
            if (obj is not LtoInfo other) return false;
            return Equals(other);
        }

        public override int GetHashCode()
        {
            return HashCode.Combine(CampaignName, Spot);
        }

        public override string ToString()
        {
            return $"{nameof(Spot)}={Spot}; {nameof(CampaignName)}={CampaignName}; ResourceType={BadgeCreative.Resource.Type}; Start={StartTime}; End={EndTime}";
        }
    }
}