﻿using System;
using JetBrains.Annotations;

namespace Magify.Rx
{
    internal sealed class Subject<T> : IObservable<T>, IDisposable
    {
        [NotNull]
        private readonly object _observerLock = new();

        private bool _isStopped;
        private bool _isDisposed;
        private Exception _lastError;
        [NotNull]
        private IObserver<T> _outObserver = EmptyObserver<T>.Instance;

        public bool HasObservers => !(_outObserver is EmptyObserver<T>) && !_isStopped && !_isDisposed;

        public void OnCompleted()
        {
            IObserver<T> old;
            lock (_observerLock)
            {
                ThrowIfDisposed();
                if (_isStopped) return;

                old = _outObserver;
                _outObserver = EmptyObserver<T>.Instance;
                _isStopped = true;
            }

            old.OnCompleted();
        }

        public void OnError(Exception error)
        {
            if (error == null) throw new ArgumentNullException(nameof(error));

            IObserver<T> old;
            lock (_observerLock)
            {
                ThrowIfDisposed();
                if (_isStopped) return;

                old = _outObserver;
                _outObserver = EmptyObserver<T>.Instance;
                _isStopped = true;
                _lastError = error;
            }

            old.OnError(error);
        }

        public void OnNext(T value)
        {
            _outObserver.OnNext(value);
        }

        public IDisposable Subscribe([NotNull] IObserver<T> observer)
        {
            Exception exception;
            lock (_observerLock)
            {
                ThrowIfDisposed();
                if (!_isStopped)
                {
                    if (_outObserver is ListObserver<T> listObserver)
                    {
                        _outObserver = listObserver.Add(observer);
                    }
                    else
                    {
                        var current = _outObserver;
                        if (current is EmptyObserver<T>)
                        {
                            _outObserver = observer;
                        }
                        else
                        {
                            _outObserver = new ListObserver<T>(new CopyOnWriteList<IObserver<T>>(new[] { current, observer }));
                        }
                    }

                    return new Subscription(this, observer);
                }

                exception = _lastError;
            }

            if (exception != null)
            {
                observer.OnError(exception);
            }
            else
            {
                observer.OnCompleted();
            }

            return EmptyDisposable.Singleton;
        }

        public void Dispose()
        {
            lock (_observerLock)
            {
                _isDisposed = true;
                _outObserver = DisposedObserver<T>.Instance;
            }
        }

        private void ThrowIfDisposed()
        {
            if (_isDisposed) throw new ObjectDisposedException("");
        }

        public bool IsRequiredSubscribeOnCurrentThread()
        {
            return false;
        }

        private class Subscription : IDisposable
        {
            [NotNull]
            private readonly object _gate = new();
            private Subject<T> _parent;
            private IObserver<T> _unsubscribeTarget;

            public Subscription(Subject<T> parent, IObserver<T> unsubscribeTarget)
            {
                this._parent = parent;
                this._unsubscribeTarget = unsubscribeTarget;
            }

            public void Dispose()
            {
                lock (_gate)
                {
                    if (_parent != null)
                    {
                        lock (_parent._observerLock)
                        {
                            if (_parent._outObserver is ListObserver<T> listObserver)
                            {
                                _parent._outObserver = listObserver.Remove(_unsubscribeTarget);
                            }
                            else
                            {
                                _parent._outObserver = EmptyObserver<T>.Instance;
                            }

                            _unsubscribeTarget = null;
                            _parent = null;
                        }
                    }
                }
            }
        }
    }
}