﻿using System;
using JetBrains.Annotations;

namespace Magify.Rx
{
    internal class WhereObservable<T> : OperatorObservableBase<T>
    {
        [NotNull]
        private readonly IObservable<T> _source;
        [NotNull]
        private readonly Func<T, bool> _predicate;

        public WhereObservable([NotNull] IObservable<T> source, [NotNull] Func<T, bool> predicate)
        {
            _source = source;
            _predicate = predicate;
        }

        // Optimize for .Where().Where()
        [NotNull]
        public IObservable<T> CombinePredicate([NotNull] Func<T, bool> combinePredicate)
        {
            return new WhereObservable<T>(_source, x => _predicate(x) && combinePredicate(x));
        }

        // Optimize for .Where().Select()
        [NotNull]
        public IObservable<TR> CombineSelector<TR>([NotNull] Func<T, TR> selector)
        {
            return new WhereSelectObservable<T, TR>(_source, _predicate, selector);
        }

        [NotNull]
        protected override IDisposable SubscribeCore([NotNull] IObserver<T> observer, [NotNull] IDisposable cancel)
        {
            return _source.Subscribe(new Where(this, observer, cancel));
        }

        private class Where : OperatorObserverBase<T, T>
        {
            [NotNull]
            private readonly WhereObservable<T> _parent;

            public Where([NotNull] WhereObservable<T> parent, [NotNull] IObserver<T> observer, [NotNull] IDisposable cancel) : base(observer, cancel)
            {
                _parent = parent;
            }

            public override void OnNext(T value)
            {
                bool isPassed;
                try
                {
                    isPassed = _parent._predicate(value);
                }
                catch (Exception ex)
                {
                    try { Observer.OnError(ex); } finally { Dispose(); }
                    return;
                }

                if (isPassed)
                {
                    Observer.OnNext(value);
                }
            }

            public override void OnError([NotNull] Exception error)
            {
                try { Observer.OnError(error); } finally { Dispose(); }
            }

            public override void OnCompleted()
            {
                try { Observer.OnCompleted(); } finally { Dispose(); }
            }
        }
    }
}