using System.Threading;
using JetBrains.Annotations;

namespace Magify
{
    /// <see cref="SyncStateResult"/>.<see cref="SyncStateResult.IsSuccess"/> will be true only if some state was restored (not empty response from server). <br/>
    public class SyncStateResult
    {
        public enum ResultKind
        {
            Success,
            NothingToRestore,
            RestoreInProgress,
            SyncIsDisabled,
            FailedToAuthorize,
            FailedToSave,
            SaveConflict,
            ForceUpdateRequested,
            FailedToRestore,
            AlreadyHasNewestProgress,
            OnlyForceRestorePossible,
            Cancelled,
        }

        public bool IsSuccess => Kind == ResultKind.Success;
        public ResultKind Kind { get; init; } = ResultKind.Success;

        /// <summary>
        /// Weight that was passed to the original request
        /// </summary>
        [CanBeNull]
        public int? Weight { get; init; }

        /// <summary>
        /// Weight of the restored progress if successful (received from server)
        /// </summary>
        [CanBeNull]
        public int? RestoredWeight { get; internal set; }

        /// <summary>
        /// True if is succeeded and magify restored itself data
        /// </summary>
        public bool InternalStateSuccessfullyRestored { get; init; }

        [CanBeNull]
        public string ErrorMessage { get; init; }

        [CanBeNull]
        public CancellationToken? CancellationToken { get; init; }

        [NotNull]
        internal static SyncStateResult Ok(int? weight, bool internalStateRestored)
        {
            return new SyncStateResult
            {
                Kind = ResultKind.Success,
                Weight = weight,
                InternalStateSuccessfullyRestored = internalStateRestored,
            };
        }

        [NotNull]
        internal static SyncStateResult NothingToRestore(int? weight)
        {
            return new SyncStateResult
            {
                Kind = ResultKind.NothingToRestore,
                Weight = weight,
            };
        }

        [NotNull]
        internal static SyncStateResult FailedToAuthorize([NotNull] string clientId)
        {
            return new SyncStateResult
            {
                Kind = ResultKind.FailedToAuthorize,
                ErrorMessage = $"Failed to authorize user with client id: {clientId}",
            };
        }

        [NotNull]
        internal static SyncStateResult RestoreInProgress()
        {
            return new SyncStateResult
            {
                Kind = ResultKind.RestoreInProgress,
                ErrorMessage = "User restore is still in progress.",
            };
        }

        [NotNull]
        internal static SyncStateResult RestoreCallbackInProgress(int? weight)
        {
            return new SyncStateResult
            {
                Kind = ResultKind.RestoreInProgress,
                ErrorMessage = "You're probably trying to call restore state during 'restore completed callback' of previous restore. It's not allowed.",
                Weight = weight,
            };
        }

        [NotNull]
        internal static SyncStateResult SyncIsDisabled()
        {
            return new SyncStateResult
            {
                Kind = ResultKind.SyncIsDisabled,
                ErrorMessage = "Sync state disabled.",
            };
        }

        [NotNull]
        internal static SyncStateResult FailedToSave([CanBeNull] string details, int? weight)
        {
            return new SyncStateResult
            {
                Kind = ResultKind.FailedToSave,
                ErrorMessage = $"Failed to save state. Details: {details}",
                Weight = weight,
            };
        }

        [NotNull]
        internal static SyncStateResult SaveConflict([CanBeNull] int? weight)
        {
            return new SyncStateResult
            {
                Kind = ResultKind.SaveConflict,
                ErrorMessage = $"Received a conflict error that occurred when requesting to save progress.",
                Weight = weight,
            };
        }

        [NotNull]
        internal static SyncStateResult ForceUpdateRequested([CanBeNull] int? weight)
        {
            return new SyncStateResult
            {
                Kind = ResultKind.ForceUpdateRequested,
                ErrorMessage = $"Force update request was received when requesting to save progress. In order to perform it you need to cause progress recovery without any weight.",
                Weight = weight,
            };
        }

        [NotNull]
        internal static SyncStateResult FailedToRestore([CanBeNull] string details, int? weight)
        {
            return new SyncStateResult
            {
                Kind = ResultKind.FailedToRestore,
                ErrorMessage = $"Failed to restore state. Details: {details}",
                Weight = weight,
            };
        }

        [NotNull]
        internal static SyncStateResult AlreadyHasNewestProgress(int? weight)
        {
            return new SyncStateResult
            {
                Kind = ResultKind.AlreadyHasNewestProgress,
                ErrorMessage = $"The progress saved on the server has less weight than the weight passed to the progress restore request ({weight}). Restore is skipped.",
                Weight = weight,
            };
        }

        [NotNull]
        internal static SyncStateResult OnlyForceRestorePossible(int? weight)
        {
            return new SyncStateResult
            {
                Kind = ResultKind.OnlyForceRestorePossible,
                ErrorMessage = $"Force update request was received. In order to perform it you need to cause progress recovery without any weight.",
                Weight = weight,
            };
        }

        [NotNull]
        internal static SyncStateResult SaveCancelled(CancellationToken cancellationToken, int? weight)
        {
            return new SyncStateResult
            {
                Kind = ResultKind.Cancelled,
                ErrorMessage = "Save state operation was cancelled.",
                CancellationToken = cancellationToken,
                Weight = weight,
            };
        }

        [NotNull]
        internal static SyncStateResult RestoreCancelled(CancellationToken cancellationToken, int? weight)
        {
            return new SyncStateResult
            {
                Kind = ResultKind.Cancelled,
                ErrorMessage = "Restore state operation was cancelled.",
                CancellationToken = cancellationToken,
                Weight = weight,
            };
        }

        [NotNull]
        internal static SyncStateResult AuthorizationCancelled(CancellationToken cancellationToken)
        {
            return new SyncStateResult
            {
                Kind = ResultKind.Cancelled,
                ErrorMessage = "Authorization operation was cancelled.",
                CancellationToken = cancellationToken,
            };
        }

        [NotNull]
        public override string ToString()
        {
            var weightString = Weight is null ? "without weight" : $"with weight {Weight}";
            var restoredWeightString = RestoredWeight is null ? string.Empty : $"Response also contains restored state weight: {RestoredWeight}";
            return Kind is ResultKind.Success
                ? $"Sync state {weightString} was successfully finished. {restoredWeightString}"
                : $"Sync state {weightString} was finished with result {Kind} and message {ErrorMessage}. {restoredWeightString}";
        }
    }
}